package ru.yandex.travel.orders.services.takeout.tasks;

import java.time.Instant;
import java.util.Collection;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.data.domain.PageRequest;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TakeoutJob;
import ru.yandex.travel.orders.repository.OrderRepository;
import ru.yandex.travel.orders.repository.TakeoutJobRepository;
import ru.yandex.travel.orders.services.takeout.AviaOrderTakeoutService;
import ru.yandex.travel.orders.services.takeout.GenericOrderTakeoutService;
import ru.yandex.travel.orders.services.takeout.HotelOrderTakeoutService;
import ru.yandex.travel.orders.services.takeout.TakeoutProperties;
import ru.yandex.travel.orders.services.takeout.TrainOrderTakeoutService;
import ru.yandex.travel.orders.takeout.proto.ETakeoutJobState;
import ru.yandex.travel.orders.takeout.proto.ETakoutJobType;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.takeout.models.TakeoutResponse;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorProperties;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.SentryHelper;

@Slf4j
@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties(TakeoutProperties.class)
public class TakeoutTaskProcessorsConfiguration {
    private final TakeoutProperties takeoutProperties;
    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;
    private final TakeoutJobRepository takeoutJobRepository;
    private final OrderRepository orderRepository;
    private final TrainOrderTakeoutService trainOrderTakeoutService;
    private final AviaOrderTakeoutService aviaOrderTakeoutService;
    private final HotelOrderTakeoutService hotelOrderTakeoutService;
    private final GenericOrderTakeoutService genericOrderTakeoutService;

    @Bean
    public TaskProcessor<UUID> takeoutJobProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return fetchJobsToProcess(getLockedTaskKeys(), maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return countJobsToProcess(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("TakeoutJobProcessor");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, this::processJob,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("TakeoutJobProcessor")
                        .daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(takeoutProperties.getTaskNumberOfItems())
                        .initialStartDelay(takeoutProperties.getTaskInitialStartDelay())
                        .scheduleRate(takeoutProperties.getTaskPeriod()).build());
    }

    @TransactionMandatory
    private List<UUID> fetchJobsToProcess(Set<UUID> active, int maxResultSize) {
        Collection<UUID> excludeFilter = active != null && !active.isEmpty() ? active :
                takeoutJobRepository.NO_EXCLUDE_IDS;
        return takeoutJobRepository.getJobsIdsByState(excludeFilter, ETakeoutJobState.TS_NEW,
                PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    private long countJobsToProcess(Set<UUID> active) {
        Collection<UUID> excludeFilter = active != null && !active.isEmpty() ? active :
                takeoutJobRepository.NO_EXCLUDE_IDS;
        return takeoutJobRepository.countJobsByState(excludeFilter, ETakeoutJobState.TS_NEW);
    }

    @TransactionMandatory
    private void processJob(UUID jobId) {
        TakeoutJob job = takeoutJobRepository.getOne(jobId);
        job.setPayload(new TakeoutResponse());
        if (job.getType() == ETakoutJobType.TT_GET) {
            try {
                if (takeoutProperties.getEnableGenericOrdersMode()) {
                    job.getPayload().setGenericOrders(genericOrderTakeoutService.getOrdersByUser(job.getUserPassportId()));
                } else {
                    job.getPayload().setTrainOrders(trainOrderTakeoutService.getOrdersByUser(job.getUserPassportId()));
                    job.getPayload().setAviaOrders(aviaOrderTakeoutService.getOrdersByUser(job.getUserPassportId()));
                    job.getPayload().setHotelOrders(hotelOrderTakeoutService.getOrdersByUser(job.getUserPassportId()));
                }
                job.getPayload().setFinishedAt(Instant.now());
                job.setState(ETakeoutJobState.TS_DONE);
            } catch (Exception ex) {
                SentryHelper.reportCrashExceptionToSentry(ex, "TakeoutJob");
                log.error("Error process takeout job {}", jobId, ex);
                job.getPayload().setFinishedAt(Instant.now());
                job.getPayload().setError(ex.getMessage());
                job.setState(ETakeoutJobState.TS_FAILED);
            }
        } else if (job.getType() == ETakoutJobType.TT_DELETE) {
            try {
                List<Order> userOrders = orderRepository.findAllOrdersOwnedByUser(job.getUserPassportId());
                userOrders.forEach(order -> order.setRemoved(true));

                job.getPayload().setFinishedAt(Instant.now());
                job.setState(ETakeoutJobState.TS_DONE);
            } catch (Exception ex) {
                SentryHelper.reportCrashExceptionToSentry(ex, "TakeoutJob");
                log.error("Error process takeout job {}", jobId, ex);
                job.getPayload().setFinishedAt(Instant.now());
                job.getPayload().setError(ex.getMessage());
                job.setState(ETakeoutJobState.TS_FAILED);
            }
        } else {
            log.error("Unknown type of takeout job {}: {}", jobId, job.getType());
        }
    }
}
