package ru.yandex.travel.orders.services.train;

import java.time.Duration;
import java.util.List;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.repository.GenericOrderRepository;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.train.proto.TOrderOfficeRefundStart;
import ru.yandex.travel.orders.workflow.train.proto.TServiceOfficeRefundStart;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.exceptions.RetryableException;
import ru.yandex.travel.workflow.single_operation.SingleOperationRunner;

@Service
@Slf4j
@RequiredArgsConstructor
public class GenericOfficeRefundStartService implements SingleOperationRunner<GenericOfficeRefundStartService.StartOfficeRefundData, String> {
    private final WorkflowMessageSender workflowMessageSender;
    private final GenericOrderRepository orderRepository;
    private final Set<EOrderState> officeRefundAllowedStates = Set.of(
            EOrderState.OS_CONFIRMED, EOrderState.OS_REFUNDED
    );
    private final Duration officeRefundRetryDelay = Duration.ofSeconds(20);

    @Override
    public Class<StartOfficeRefundData> getInputClass() {
        return StartOfficeRefundData.class;
    }

    @Override
    public String runOperation(StartOfficeRefundData data) {
        GenericOrder order = orderRepository.getOne(data.orderId);
        try (var ignored = NestedMdc.forEntity(order)) {
            if (order.isUserActionScheduled() || !officeRefundAllowedStates.contains(order.getEntityState())) {
                var message = "Can't start office refund because " +
                        (order.isUserActionScheduled() ? "UserActionScheduled" : order.getEntityState().name());
                throw new RetryableException(message, officeRefundRetryDelay);
            }

            Preconditions.checkArgument(
                    data.getServices().stream().allMatch(x -> x.blankIds == null),
                    "blankIds shouldn't be used in Order %s",
                    data.orderId
            );

            order.toggleUserActionScheduled(true);
            List<TServiceOfficeRefundStart> services = data.getServices().stream()
                    .map(x -> TServiceOfficeRefundStart.newBuilder()
                            .setServiceId(x.serviceId.toString())
                            .addAllRefundOperationIds(x.refundOperationIds).build())
                    .collect(Collectors.toList());
            workflowMessageSender.scheduleEvent(order.getWorkflow().getId(),
                    TOrderOfficeRefundStart.newBuilder().addAllServices(services).build());
            return null;
        }
    }

    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class StartOfficeRefundData {
        UUID orderId;
        List<Service> services;
    }

    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class Service {
        UUID serviceId;
        Set<Integer> refundOperationIds;

        @Deprecated(forRemoval = true)
        Set<Integer> blankIds;
    }
}
