package ru.yandex.travel.orders.services.train;

import java.util.ArrayList;
import java.util.Comparator;
import java.util.List;

import com.google.common.base.Strings;
import lombok.AccessLevel;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoDataProvider;
import ru.yandex.travel.orders.workflows.orderitem.train.ImHelpers;
import ru.yandex.travel.train.model.AdditionalPlaceRequirements;
import ru.yandex.travel.train.model.CabinGenderKind;
import ru.yandex.travel.train.model.CabinPlaceDemands;
import ru.yandex.travel.train.model.CarStorey;
import ru.yandex.travel.train.model.DocumentType;
import ru.yandex.travel.train.model.PassengerCategory;
import ru.yandex.travel.train.model.RailwayBonusCard;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.partners.im.model.OrderFullCustomerRequest;
import ru.yandex.travel.train.partners.im.model.PlaceRange;
import ru.yandex.travel.train.partners.im.model.RailwayCardInfo;
import ru.yandex.travel.train.partners.im.model.RailwayPassengerRequest;
import ru.yandex.travel.train.partners.im.model.RailwayReservationRequest;
import ru.yandex.travel.train.partners.im.model.ReservationCreateRequest;

@Slf4j
@RequiredArgsConstructor(access = AccessLevel.PRIVATE)
public class ImReservationManager {
    private final TrainTariffInfoDataProvider trainTariffInfoDataProvider;
    private final TrainOrderItem orderItem;
    private final List<TrainOrderItem> slaves;

    private int itemIndexCounter;
    private List<RailwayReservationRequest> items;

    public static ReservationCreateRequest createRequest(
            TrainTariffInfoDataProvider trainTariffInfoDataProvider,
            TrainOrderItem orderItem,
            List<TrainOrderItem> slaves
    ) {
        return new ImReservationManager(trainTariffInfoDataProvider, orderItem, slaves).createRequest();
    }

    private ReservationCreateRequest createRequest() {
        itemIndexCounter = 0;
        items = new ArrayList<>();
        List<OrderFullCustomerRequest> customers = new ArrayList<>();
        var request = new ReservationCreateRequest();
        var payload = orderItem.getPayload();
        List<TrainPassenger> passengers = payload.getPassengers();
        for (int i = 0, passengersSize = passengers.size(); i < passengersSize; i++) {
            TrainPassenger passenger = passengers.get(i);
            var customer = new OrderFullCustomerRequest();
            customers.add(customer);
            customer.setFirstName(passenger.getFirstName());
            customer.setLastName(passenger.getLastName());
            customer.setMiddleName(passenger.getPatronymic());
            customer.setSex(passenger.getSex());
            customer.setBirthday(passenger.getBirthday().atStartOfDay());
            customer.setCitizenshipCode(passenger.getCitizenshipCode());
            customer.setDocumentType(passenger.getDocumentType());
            customer.setDocumentNumber(passenger.getDocumentNumber());
            customer.setIndex(i);
        }
        createServiceReservationItems(orderItem);
        for (var slave : slaves) {
            createServiceReservationItems(slave);
        }
        request.setCustomers(customers);
        request.setReservationItems(items);
        return request;
    }

    private void createServiceReservationItems(TrainOrderItem orderItem) {
        var payload = orderItem.getPayload();
        List<RailwayReservationRequest> serviceReservationItems = new ArrayList<>();

        List<TrainPassenger> passengers = payload.getPassengers();
        for (int i = 0, passengersSize = passengers.size(); i < passengersSize; i++) {
            TrainPassenger passenger = passengers.get(i);
            RailwayPassengerRequest requestPassenger = createReservationRequestPassenger(passenger,
                    orderItem.getOrder().getEmail(), i);
            RailwayReservationRequest reserveItem = getOrCreateReservationRequestItemForPassenger(payload,
                    serviceReservationItems,
                    passenger);
            passenger.setPartnerItemIndex(reserveItem.getIndex());
            reserveItem.getPassengers().add(requestPassenger);
        }

        for (var reserveItem : serviceReservationItems) {
            if (payload.isCppk()) {
                fixCppkRequestItem(reserveItem);
            }
        }
        items.addAll(serviceReservationItems);
    }

    private RailwayPassengerRequest createReservationRequestPassenger(TrainPassenger passenger, String defaultEmail,
                                                                      Integer customerIndex) {
        var requestPassenger = new RailwayPassengerRequest();
        requestPassenger.setOrderCustomerIndex(customerIndex);
        requestPassenger.setCategory(passenger.getCategory());
        requestPassenger.setPreferredAdultTariffType(
                trainTariffInfoDataProvider.getImRequestCode(passenger.getTariffCodeWithFallback()));
        requestPassenger.setNonRefundableTariff(passenger.isNonRefundableTariff());

        if (passenger.getBonusCards() != null) {
            requestPassenger.setRailwayBonusCards(new ArrayList<>());
            for (RailwayBonusCard card : passenger.getBonusCards()) {
                var requestCard = new RailwayCardInfo();
                requestPassenger.getRailwayBonusCards().add(requestCard);
                requestCard.setCardNumber(card.getCardNumber());
                requestCard.setCardType(card.getCardType());
            }
        }

        if (passenger.isUsePhoneForReservation() && !Strings.isNullOrEmpty(passenger.getPhone())) {
            boolean isForeignDocument = passenger.getDocumentType() == DocumentType.FOREIGN_DOCUMENT;
            String normalizedPhone = ImHelpers.formatImNonContactPhone(passenger.getPhone(), isForeignDocument);
            log.info("Phone normalization: '{}' -> '{}'", passenger.getPhone(), normalizedPhone);
            if (normalizedPhone != null) {
                requestPassenger.setPhone(normalizedPhone);
            } else {
                log.info("The provided phone can't be used for IM notifications, ignoring it");
                passenger.setUsePhoneForReservation(false);
            }
        }

        if (passenger.isUseEmailForReservation()) {
            if (Strings.isNullOrEmpty(passenger.getEmail())) {
                requestPassenger.setContactEmailOrPhone(defaultEmail);
            } else {
                requestPassenger.setContactEmailOrPhone(passenger.getEmail());
            }
        }
        return requestPassenger;
    }

    private RailwayReservationRequest getOrCreateReservationRequestItemForPassenger(
            TrainReservation payload, List<RailwayReservationRequest> serviceReservationItems,
            TrainPassenger passenger) {
        var reservationData = payload.getReservationRequestData();
        if (payload.isSeparatePassengersReserving()) {
            if (passenger.getCategory() == PassengerCategory.BABY) {
                return serviceReservationItems.stream()
                        .filter(x -> x.getPassengers().stream().filter(p -> p.getCategory() == PassengerCategory.ADULT).count() >
                                x.getPassengers().stream().filter(p -> p.getCategory() == PassengerCategory.BABY).count())
                        .findFirst()
                        .orElseThrow(() -> new RuntimeException("Can't find service reservation item suitable for a " +
                                "baby"));
            } else {
                var reserveItem = createReservationRequestItem(payload);
                if (passenger.getRequestedPlaces() != null && passenger.getRequestedPlaces().size() > 0) {
                    reserveItem.setPlaceRange(new PlaceRange());
                    reserveItem.getPlaceRange().setFrom(passenger.getRequestedPlaces().stream().min(Comparator.naturalOrder()).get());
                    reserveItem.getPlaceRange().setTo(passenger.getRequestedPlaces().stream().max(Comparator.naturalOrder()).get());
                }
                serviceReservationItems.add(reserveItem);
                return reserveItem;
            }
        } else if (serviceReservationItems.size() == 0) {
            var reserveItem = createReservationRequestItem(payload);
            reserveItem.setLowerPlaceQuantity(reservationData.getLowerPlaceQuantity());
            reserveItem.setUpperPlaceQuantity(reservationData.getUpperPlaceQuantity());
            if (reservationData.getPlaceNumberFrom() != null) {
                reserveItem.setPlaceRange(new PlaceRange());
                reserveItem.getPlaceRange().setFrom(reservationData.getPlaceNumberFrom());
                reserveItem.getPlaceRange().setTo(reservationData.getPlaceNumberTo());
            }
            serviceReservationItems.add(reserveItem);
            return reserveItem;
        } else {
            return serviceReservationItems.get(0);
        }
    }

    private RailwayReservationRequest createReservationRequestItem(TrainReservation payload) {
        var reservationData = payload.getReservationRequestData();
        var reserveItem = new RailwayReservationRequest();
        reserveItem.setIndex(itemIndexCounter++);
        reserveItem.setProvider(payload.getProvider());
        reserveItem.setSetElectronicRegistration(reservationData.isElectronicRegistrationEnabled());
        reserveItem.setAdditionalPlaceRequirements(reservationData.getAdditionalPlaceRequirements());
        reserveItem.setBedding(reservationData.getBedding());
        reserveItem.setCabinGenderKind(reservationData.getCabinGenderKind());
        reserveItem.setCabinPlaceDemands(reservationData.getCabinPlaceDemands());
        reserveItem.setCarNumber(reservationData.getCarNumber());
        reserveItem.setCarStorey(reservationData.getCarStorey());
        reserveItem.setCarType(reservationData.getCarType());
        reserveItem.setDepartureDate(ImHelpers.toLocalDateTime(reservationData.getDepartureTime(),
                payload.getStationFromRailwayTimezone()));
        reserveItem.setOriginCode(reservationData.getStationFromCode());
        reserveItem.setDestinationCode(reservationData.getStationToCode());
        reserveItem.setGiveAdditionalTariffForChildIfPossible(reservationData.isGiveChildWithoutPlace());
        reserveItem.setInternationalServiceClass(reservationData.getInternationalServiceClass());
        reserveItem.setTrainNumber(reservationData.getTrainNumber());
        reserveItem.setServiceClass(reservationData.getServiceClass());
        reserveItem.setPassengers(new ArrayList<>());
        return reserveItem;
    }

    private void fixCppkRequestItem(RailwayReservationRequest item) {
        item.setGiveAdditionalTariffForChildIfPossible(false);
        item.setOnRequestMeal(false);
        item.setCabinGenderKind(CabinGenderKind.NO_VALUE);
        item.setCabinPlaceDemands(CabinPlaceDemands.NO_VALUE);
        item.setClientCharge(null);
        item.setInternationalServiceClass(null);
        item.setSpecialPlacesDemand("NoValue");
        item.setCarStorey(CarStorey.NO_VALUE);
        item.setAdditionalPlaceRequirements(AdditionalPlaceRequirements.NO_VALUE);
        for (var passenger : item.getPassengers()) {
            passenger.setNonRefundableTariff(false);
            passenger.setInvalid(false);
            passenger.setDisabledPersonId(null);
            passenger.setTransitDocument("NoValue");
            // если вдруг начнем использовать поле ИМа TransportationRequirement, то обнулить его для ЦППК
            //passenger.setTransportationRequirement(null);
            passenger.setRailwayBonusCards(null);
        }
    }

}
