package ru.yandex.travel.orders.services.train;

import java.math.BigDecimal;
import java.time.Duration;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.NoArgsConstructor;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.mock.MockTrainOrder;
import ru.yandex.travel.orders.entities.mock.MockTrainOrderItem;
import ru.yandex.travel.orders.repository.GenericOrderRepository;
import ru.yandex.travel.orders.repository.mock.MockTrainOrderItemRepository;
import ru.yandex.travel.orders.repository.mock.MockTrainOrderRepository;
import ru.yandex.travel.orders.services.mock.MockImResponseGenerator;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.train.proto.TOrderOfficeRefundStart;
import ru.yandex.travel.orders.workflow.train.proto.TServiceOfficeRefundStart;
import ru.yandex.travel.train.partners.im.model.ImBlankStatus;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemBlank;
import ru.yandex.travel.workflow.WorkflowMessageSender;
import ru.yandex.travel.workflow.exceptions.RetryableException;
import ru.yandex.travel.workflow.single_operation.SingleOperationRunner;

@Service
@Slf4j
@RequiredArgsConstructor
public class MockImGenericOfficeRefundStartService
        implements SingleOperationRunner<MockImGenericOfficeRefundStartService.StartOfficeRefundData, String> {
    private final WorkflowMessageSender workflowMessageSender;
    private final GenericOrderRepository orderRepository;
    private final MockImResponseGenerator mockImResponseGenerator;
    private final MockTrainOrderRepository mockTrainOrderRepository;
    private final MockTrainOrderItemRepository mockTrainOrderItemRepository;

    @Override
    public Class<StartOfficeRefundData> getInputClass() {
        return StartOfficeRefundData.class;
    }

    @Override
    public String runOperation(StartOfficeRefundData data) {
        GenericOrder order = orderRepository.getOne(data.orderId);
        try (var ignored = NestedMdc.forEntity(order)) {
            log.info("order.getEntityState() == {}", order.getEntityState());
            if (order.isUserActionScheduled() || (
                    order.getEntityState() != EOrderState.OS_CONFIRMED
                            && order.getEntityState() != EOrderState.OS_REFUNDED
            )) {
                throw new RetryableException("Can not start office refund in this order state",
                        Duration.of(15, ChronoUnit.SECONDS));
            }
            MockTrainOrder mockOrder = mockTrainOrderRepository.getOne(data.getMockImOrderId());
            Set<Integer> refundOperationIds = new HashSet<>();
            for (var service : data.services) {
                MockTrainOrderItem mockOrderItem = mockOrder.getItems().stream()
                        .filter(x -> x.getId() == service.mockImOrderItemId).findFirst().orElseThrow();
                MockTrainOrderItem refundItem = new MockTrainOrderItem();
                refundItem.setId(mockTrainOrderRepository.getNextOrderItemId());
                refundItem.setType(MockTrainOrderItem.MockTrainOrderItemType.REFUND);
                refundItem.setOrder(mockOrder);
                refundItem.setTestContext(mockOrderItem.getTestContext());
                refundItem.setItemData(new MockTrainOrderItem.MockTrainOrderItemData(
                        mockImResponseGenerator.createRefundRailwayItemResponse(refundItem.getId(),
                                mockOrderItem.getItemData().getItemInfo(),
                                null, true, service.blankIdsWithAmounts,
                                true)));
                mockTrainOrderItemRepository.save(refundItem);
                refundOperationIds.add(refundItem.getId());

                for (OrderItemBlank b : mockOrderItem.getItemData().getItemInfo().getOrderItemBlanks()) {
                    if (service.blankIdsWithAmounts.containsKey(b.getOrderItemBlankId())) {
                        b.setBlankStatus(ImBlankStatus.REFUNDED);
                    }
                }
            }
            Preconditions.checkArgument(order.getPublicType() == EOrderType.OT_GENERIC, "Order should be generic");
            order.toggleUserActionScheduled(true);
            Map<Integer, UUID> imItemIdToServiceId = new HashMap<>();
            for (var orderItem : OrderCompatibilityUtils.getTrainOrderItems(order)) {
                for (var passenger : orderItem.getPayload().getPassengers()) {
                    imItemIdToServiceId.put(passenger.getTicket().getPartnerBuyOperationId(), orderItem.getId());
                }
            }
            List<TServiceOfficeRefundStart> services = data.getServices().stream()
                    .map(x -> TServiceOfficeRefundStart.newBuilder()
                            .setServiceId(imItemIdToServiceId.get(x.mockImOrderItemId).toString())
                            .addAllRefundOperationIds(refundOperationIds)
                            .build())
                    .collect(Collectors.toList());
            workflowMessageSender.scheduleEvent(order.getWorkflow().getId(),
                    TOrderOfficeRefundStart.newBuilder().addAllServices(services).build());
            return null;
        }
    }

    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class StartOfficeRefundData {
        int mockImOrderId;
        UUID orderId;
        List<Service> services;
    }

    @Data
    @NoArgsConstructor
    @AllArgsConstructor
    public static class Service {
        int mockImOrderItemId;
        Map<Integer, Optional<BigDecimal>> blankIdsWithAmounts;
    }
}
