package ru.yandex.travel.orders.services.train;

import java.util.Comparator;
import java.util.Map;
import java.util.Optional;

import com.google.common.base.Preconditions;
import com.netflix.concurrency.limits.Limiter;
import com.netflix.concurrency.limits.limiter.SimpleLimiter;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.Error;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TPartnerInfoUpdated;
import ru.yandex.travel.orders.workflows.orderitem.train.ImHelpers;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainWorkflowProperties;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicket;
import ru.yandex.travel.train.partners.im.ImClient;
import ru.yandex.travel.train.partners.im.ImClientRetryableException;
import ru.yandex.travel.train.partners.im.model.PendingElectronicRegistration;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderInfoResponse;
import ru.yandex.travel.train.partners.im.model.orderinfo.OrderItemBlank;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Service
@Slf4j
@RequiredArgsConstructor
public class OrderItemPayloadService {
    private final ImClientProvider imClientProvider;
    private final WorkflowMessageSender workflowMessageSender;
    private final TrainWorkflowProperties trainWorkflowProperties;
    private final SimpleLimiter<Void> updateOnTheFlyLimiter;

    public static void mergeReservation(
            TrainReservation trainReservation, OrderInfoResponse orderInfoResponse) {
        Map<Integer, OrderItemBlank> blanksById = orderInfoResponse.getBlankIdToBlankMap();
        var firstBuyItem = orderInfoResponse
                .findItem(trainReservation.getPartnerBuyOperationIds().stream().min(Comparator.naturalOrder()).get());
        trainReservation.setBoardingSystemType(firstBuyItem.getBoardingSystemType());
        for (TrainPassenger passenger : trainReservation.getPassengers()) {
            TrainTicket ticket = passenger.getTicket();
            OrderItemBlank blank = blanksById.get(ticket.getBlankId());

            ticket.setImBlankStatus(blank.getBlankStatus());
            ticket.setPendingElectronicRegistration(
                    blank.getPendingElectronicRegistration() == PendingElectronicRegistration.TO_CANCEL);
            ticket.setCanChangeElectronicRegistrationTill(ImHelpers.fromLocalDateTime(
                    blank.getElectronicRegistrationExpirationDateTime(), trainReservation.getStationFromRailwayTimezone()));
            ticket.setCanReturnTill(ImHelpers.fromLocalDateTime(
                    blank.getOnlineTicketReturnExpirationDateTime(), trainReservation.getStationFromRailwayTimezone()));
        }
    }

    public Object getPayload(OrderItem orderItem, boolean updateOrderOnTheFly) {
        Object payload = orderItem.getPayload();
        if (payload == null) {
            return null;
        }
        if (updateOrderOnTheFly && orderItem.getPublicType() == EServiceType.PT_TRAIN) {
            payload = getActualizedPayload((TrainOrderItem) orderItem);
        }
        return payload;
    }

    private TrainReservation getActualizedPayload(TrainOrderItem orderItem) {
        if (orderItem.getItemState() != EOrderItemState.IS_CONFIRMED &&
                orderItem.getItemState() != EOrderItemState.IS_REFUNDED) {
            return orderItem.getPayload();
        }

        var workflowId = orderItem.getWorkflow().getId();
        var payloadCopy = ProtoUtils.fromTJson(ProtoUtils.toTJson(orderItem.getPayload()), TrainReservation.class);

        OrderInfoResponse orderInfoResponse;

        Optional<Limiter.Listener> optionalListener = updateOnTheFlyLimiter.acquire(null);
        try {
            if (optionalListener.isEmpty()) {
                log.warn("Couldn't acquire token from limiter. Call to IM overloaded. Do not return order info");
                throw Error.with(EErrorCode.EC_CALL_TO_IM_OVERLOADED, "Update on the fly limit exceeded").toEx();
            }
            ImClient imClient = imClientProvider.getImClientForOrderItem(orderItem);
            for (var imOrderItemId : payloadCopy.getPartnerBuyOperationIds()) {
                imClient.updateBlanks(imOrderItemId, trainWorkflowProperties.getUpdateOnTheFlyUpdateBlanksTimeout());
            }
            orderInfoResponse = imClient.orderInfo(payloadCopy.getPartnerOrderId(),
                    trainWorkflowProperties.getUpdateOnTheFlyOrderInfoTimeout());
        } catch (ImClientRetryableException e) {
            log.error("Retryable error occurred while refreshing order info", e);
            throw Error.with(EErrorCode.EC_IM_RETRYABLE_ERROR, e.getMessage()).withCause(e).toEx();
        } finally {
            // if we got a listener on acquire then we must release it.
            // always assume call to be a success, as there's a retry logic and we want the following behaviour:
            // we use AIMD limit here and on timeouts it'll cut the number of concurrent calls so if IM is timing out
            // we'll reduce the number of concurrent calls, and then raising the number of concurrent calls when
            // timings stabilize
            optionalListener.ifPresent(Limiter.Listener::onSuccess);
        }


        Preconditions.checkNotNull(orderInfoResponse,
                "Order info response must be initialized with correct info");

        mergeReservation(payloadCopy, orderInfoResponse);

        var message = TPartnerInfoUpdated.newBuilder()
                .setPayload(ProtoUtils.toTJson(orderInfoResponse)).build();

        workflowMessageSender.scheduleEvent(workflowId, message);

        return payloadCopy;
    }
}
