package ru.yandex.travel.orders.services.train;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.repository.TrainOrderItemRepository;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.train.model.Insurance;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainPlace;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicket;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

@Service
@Slf4j
@RequiredArgsConstructor
public class RebookingService {
    private final TrainOrderItemRepository trainOrderItemRepository;
    private final WorkflowRepository workflowRepository;
    private final TrainMeters trainMeters;

    public OrderItem rebookItem(Order order, TrainOrderItem source) {
        TrainReservation payloadCopy = ProtoUtils.fromTJson(ProtoUtils.toTJson(source.getPayload()), TrainReservation.class);
        TrainOrderItem copy = new TrainOrderItem();
        copy.setReservation(payloadCopy);
        copy.setArchived(true);
        copy.setItemNumber(source.getItemNumber());
        copy.setState(source.getState());
        copy.setExpired(source.isExpired());
        copy.setExpiresAt(source.getExpiresAt());
        Workflow oldWorkflow = source.getWorkflow();
        copy.setWorkflow(oldWorkflow);
        copy = trainOrderItemRepository.save(copy);
        oldWorkflow.setEntityId(copy.getId());
        order.addOrderItem(copy);

        TrainReservation payload = source.getPayload();
        payload.setIsRebookingFor(copy.getId());
        for (var p : payload.getPassengers()) {
            p.setTicket(null);
            p.setInsurance(null);
            p.setCustomerId(0);
        }
        payload.setReservationNumber(null);
        payload.setPartnerOrderId(0);
        payload.setPartnerBuyOperationId(0);
        payload.setCanChangeElectronicRegistrationTill(null);
        payload.setTargetInsuranceStatus(payload.getInsuranceStatus());
        payload.setInsuranceStatus(payload.getInsuranceStatus() == InsuranceStatus.DISABLED ?
                InsuranceStatus.DISABLED : InsuranceStatus.NEW);
        if (payload.getReservationRequestData().getPlaceNumberFrom() == null) {
            try {
                List<Integer> placeList = payloadCopy.getPassengers().stream()
                        .flatMap(p -> p.getTicket().getPlaces().stream()).map(p -> Integer.parseInt(p.getNumber()))
                        .collect(Collectors.toList());
                if (placeList.size() > 0) {
                    Integer placeNumberFrom = placeList.stream().min(Integer::compareTo).get();
                    Integer placeNumberTo = placeList.stream().max(Integer::compareTo).get();
                    payload.getReservationRequestData().setPlaceNumberFrom(placeNumberFrom);
                    payload.getReservationRequestData().setPlaceNumberTo(placeNumberTo);
                }
            } catch (Exception ex) {
                log.error("Error restore place number range from tickets", ex);
            }
        }
        source.setState(EOrderItemState.IS_NEW);
        source.setExpired(false);
        source.setExpiresAt(null);
        Workflow newWorkflow = Workflow.createWorkflowForEntity(source, order.getWorkflow().getId());
        workflowRepository.save(newWorkflow);
        trainMeters.getTrainOrdersRebookingCreated().increment();
        return source;
    }

    public boolean checkRebookingValid(TrainOrderItem orderItem, TrainOrderItem oldItem) {
        List<TrainPassenger> oldPassengers = oldItem.getPayload().getPassengers();
        List<TrainPassenger> passengers = orderItem.getPayload().getPassengers();
        List<String> validationErrors = new ArrayList<>();
        for (int i = 0; i < passengers.size(); i++) {
            TrainPassenger passenger = passengers.get(i);
            TrainPassenger oldPassenger = oldPassengers.get(i);
            TrainTicket ticket = passenger.getTicket();
            TrainTicket oldTicket = oldPassenger.getTicket();
            var places = ticket.getPlaces().stream().map(TrainPlace::getNumber).collect(Collectors.toSet());
            var oldPlaces = oldTicket.getPlaces().stream().map(TrainPlace::getNumber).collect(Collectors.toSet());
            if (!places.equals(oldPlaces)) {
                validationErrors.add(String.format("ticket %s places %s -> %s", i, oldPlaces, places));
            }
            checkEqualCost(passenger, oldPassenger, x -> x.getTicket().getFeeAmount(), "ticket fee", validationErrors);
            checkEqualCost(passenger, oldPassenger, x -> x.getTicket().getServiceAmount(), "ticket service amount", validationErrors);
            checkEqualCost(passenger, oldPassenger, x -> x.getTicket().getTariffAmount(), "ticket tariff amount", validationErrors);

            Insurance oldInsurance = oldPassenger.getInsurance();
            if (orderItem.getPayload().getTargetInsuranceStatus() == InsuranceStatus.CHECKED_OUT
                    && oldInsurance != null && oldInsurance.getFiscalItemInternalId() != null) {
                checkEqualCost(passenger, oldPassenger, x -> x.getInsurance().getAmount(), "insurance amount", validationErrors);
            }
        }
        if (validationErrors.size() > 0) {
            String errorMsg = "Rebooking failed. Reservation changed after rebooking: " + String.join("; ", validationErrors);
            log.error(errorMsg);
            trainMeters.getTrainOrdersRebookingInvalid().increment();
            return false;
        }
        return true;
    }

    private void checkEqualCost(TrainPassenger passenger, TrainPassenger oldPassenger,
                                Function<TrainPassenger, Money> getter,
                                String label, List<String> validationErrors) {
        Money cost = getter.apply(passenger);
        Money oldCost = getter.apply(oldPassenger);
        if (!cost.isEqualTo(oldCost)) {
            validationErrors.add(String.format("%s %s -> %s", label, oldCost, cost));
        }
    }

    public void bindFiscalItems(TrainOrderItem orderItem, TrainOrderItem oldOrderItem) {
        List<TrainPassenger> passengers = orderItem.getPayload().getPassengers();
        List<TrainPassenger> oldPassengers = oldOrderItem.getPayload().getPassengers();
        for (int i = 0; i < passengers.size(); i++) {
            TrainPassenger passenger = passengers.get(i);
            TrainPassenger oldPassenger = oldPassengers.get(i);
            TrainTicket ticket = passenger.getTicket();
            TrainTicket oldTicket = oldPassenger.getTicket();
            ticket.setTariffFiscalItemInternalId(oldTicket.getTariffFiscalItemInternalId());
            ticket.setServiceFiscalItemInternalId(oldTicket.getServiceFiscalItemInternalId());
            ticket.setFeeFiscalItemInternalId(oldTicket.getFeeFiscalItemInternalId());

            Insurance insurance = passenger.getInsurance();
            Insurance oldInsurance = oldPassenger.getInsurance();
            if (oldInsurance != null && oldInsurance.getFiscalItemInternalId() != null) {
                insurance.setFiscalItemInternalId(oldInsurance.getFiscalItemInternalId());
            }
        }
    }
}
