package ru.yandex.travel.orders.services.train;

import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.TrainDiscount;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.repository.TrainDiscountRepository;
import ru.yandex.travel.orders.services.train.tariffinfo.TrainTariffInfoDataProvider;
import ru.yandex.travel.orders.workflows.orderitem.train.TrainOrderItemHelpers;
import ru.yandex.travel.train.model.TrainPassenger;
import ru.yandex.travel.train.model.TrainReservation;

@Service
@Slf4j
@RequiredArgsConstructor
public class TrainDiscountService {
    private final TrainDiscountRepository trainDiscountRepository;

    public void checkAndStoreUniqueDiscounts(TrainOrderItem orderItem) {
        TrainReservation payload = orderItem.getPayload();
        List<TrainPassenger> passengersWithDiscount = passengersWithDiscount(payload.getPassengers());

        if (!passengersWithDiscount.isEmpty()) {
            var departure = payload.getReservationRequestData().getDepartureTime();
            var trainNumber = payload.getReservationRequestData().getTrainNumber();
            Set<String> documentNumbersWithDiscount = trainDiscountRepository.findDiscounts(departure, trainNumber,
                    passengersWithDiscount.stream().map(TrainPassenger::getDocumentNumber).collect(Collectors.toList()))
                    .stream().map(TrainDiscount::getDocumentNumber).collect(Collectors.toUnmodifiableSet());

            for (TrainPassenger passenger: passengersWithDiscount) {
                if (documentNumbersWithDiscount.contains(passenger.getDocumentNumber())) {
                    passenger.setDiscountDenied(true);
                    passenger.setTariffCode(TrainTariffInfoDataProvider.FULL_TARIFF_CODE);
                } else {
                    var trainDiscount = new TrainDiscount();
                    trainDiscount.setDepartureTime(departure);
                    trainDiscount.setTrainNumber(trainNumber);
                    trainDiscount.setDocumentNumber(passenger.getDocumentNumber());
                    trainDiscount.setTariffCode(passenger.getTariffCodeWithFallback());
                    trainDiscount.setOrderItem(orderItem);
                    trainDiscountRepository.save(trainDiscount);
                }
            }
        }
    }

    public void deleteDiscountsForOrder(TrainOrderItem orderItem) {
        TrainReservation payload = orderItem.getPayload();
        List<TrainPassenger> passengersWithDiscount = passengersWithDiscount(payload.getPassengers());
        if (!passengersWithDiscount.isEmpty()) {
            trainDiscountRepository.deleteAllByOrderItemId(orderItem.getId());
        }
    }

    public void deleteDiscountsForOrderByBlanks(TrainOrderItem orderItem, Set<Integer> blankIds) {
        List<TrainPassenger> passengersWithDiscount = passengersWithDiscount(
                TrainOrderItemHelpers.getPassengersByBlankIds(orderItem, blankIds));
        if (!passengersWithDiscount.isEmpty()) {
            for (var passenger : passengersWithDiscount) {
                trainDiscountRepository.deleteAllByOrderItemIdAndDocumentNumber(orderItem.getId(), passenger.getDocumentNumber());
            }
        }
    }

    private List<TrainPassenger> passengersWithDiscount(List<TrainPassenger> passengers) {
        return passengers.stream()
                .filter(p -> !TrainTariffInfoDataProvider.FULL_TARIFF_CODE.equals(p.getTariffCodeWithFallback()))
                .collect(Collectors.toList());
    }
}
