package ru.yandex.travel.orders.services.train;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.stereotype.Service;

import ru.yandex.bolts.collection.Tuple2;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.orders.cpa.TTrainRefundExportLogRecord;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.train.model.refund.PassengerRefundInfo;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.AfterCommitEventLogging;

@Service
@Slf4j
public class TrainRefundLogService {
    public static final Logger TRAIN_REFUND_EVENTS_LOGGER =
            LoggerFactory.getLogger("ru.yandex.travel.orders.workflows.train.TrainRefundEventsLogger");

    @TransactionMandatory
    public void logRefund(Order order, List<TrainTicketRefund> ticketRefunds) {
        Preconditions.checkState(ticketRefunds.size() > 0, "Can not log refund without ticket refunds");
        TTrainRefundExportLogRecord.Builder builder = TTrainRefundExportLogRecord.newBuilder();

        builder.setRefundedAt(Instant.now().getEpochSecond());
        builder.setOrderId(order.getId().toString());
        builder.setOrderPrettyId(order.getPrettyId());
        List<UUID> orderRefundIds = ticketRefunds.stream().map(x -> x.getOrderRefund().getId())
                .distinct().collect(Collectors.toList());
        Preconditions.checkArgument(orderRefundIds.size() == 1, "Ticket refunds contains different orderRefunds");
        builder.setOrderRefundId(orderRefundIds.get(0).toString());

        int numberOfTickets = ticketRefunds.stream()
                .map(x -> x.getPayload().getRefundedItems().size())
                .reduce(Integer::sum).orElse(0);

        Money refundTicketAmount = Money.zero(ProtoCurrencyUnit.RUB);
        Money refundFeeAmount = Money.zero(ProtoCurrencyUnit.RUB);
        Money refundInsuranceAmount = Money.zero(ProtoCurrencyUnit.RUB);
        Money partnerRefundFeeAmount = Money.zero(ProtoCurrencyUnit.RUB);

        Map<Tuple2<Integer, Integer>, Money> partnerRefundByPassengerBlank = ticketRefunds.stream()
                .flatMap(x -> x.getOrderItem().getPayload().getPassengers().stream())
                .collect(Collectors.toUnmodifiableMap(
                        tp -> Tuple2.tuple(tp.getCustomerId(), tp.getTicket().getBlankId()),
                        tp -> tp.getTicket().getPartnerRefundFee()
                ));

        for (TrainTicketRefund ticketRefund : ticketRefunds) {
            for (PassengerRefundInfo pri : ticketRefund.getPayload().getRefundedItems()) {
                refundTicketAmount = refundTicketAmount.add(
                        pri.getActualRefundTicketAmount() != null ? pri.getActualRefundTicketAmount() :
                                pri.getCalculatedRefundTicketAmount()
                );
                refundFeeAmount = refundFeeAmount.add(pri.getCalculatedRefundFeeAmount());
                refundInsuranceAmount = refundInsuranceAmount.add(pri.getCalculatedRefundInsuranceAmount());
                Money partnerRefundFeeForBlank = partnerRefundByPassengerBlank.get(Tuple2.tuple(pri.getCustomerId(),
                        pri.getBlankId()));
                if (partnerRefundFeeForBlank != null) {
                    partnerRefundFeeAmount = partnerRefundFeeAmount.add(partnerRefundFeeForBlank);
                }
            }
        }

        builder.setNumberOfTickets(numberOfTickets);
        builder.setRefundTicketMoney(convertMoney(refundTicketAmount));
        builder.setRefundFeeMoney(convertMoney(refundFeeAmount));
        builder.setRefundInsuranceMoney(convertMoney(refundInsuranceAmount));
        builder.setPartnerRefundFeeMoney(convertMoney(partnerRefundFeeAmount));

        AfterCommitEventLogging.logEvent(TRAIN_REFUND_EVENTS_LOGGER, builder.build(), true);
    }

    private TTrainRefundExportLogRecord.TPriceExportInfo convertMoney(Money money) {
        return TTrainRefundExportLogRecord.TPriceExportInfo.newBuilder()
                .setCurrency(money.getCurrency().getCurrencyCode())
                .setAmount(money.getNumberStripped().doubleValue())
                .build();
    }
}
