package ru.yandex.travel.orders.services.train.bandit;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Objects;

import io.grpc.Channel;
import io.grpc.ClientInterceptor;
import io.grpc.ManagedChannelBuilder;
import io.opentracing.Tracer;
import lombok.SneakyThrows;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TracerClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TvmHeaderClientHelper;
import ru.yandex.travel.train_bandit_api.proto.BanditApiServiceV1Grpc;
import ru.yandex.travel.tvm.TvmWrapper;

@Configuration
@EnableConfigurationProperties(TrainBanditProperties.class)
public class TrainBanditConfiguration {

    @Bean
    @SneakyThrows
    public BanditApiServiceV1Grpc.BanditApiServiceV1BlockingStub BanditApiServiceV1BlockingStub(
            TrainBanditProperties properties, Tracer tracer,
            @Autowired(required = false) TvmWrapper tvm) {
        String clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
        List<ClientInterceptor> interceptors = new ArrayList<>(4);
        interceptors.add(new LoggingClientInterceptor(
                clientFqdn, properties.getTarget(), Collections.emptySet()
        ));
        interceptors.add(new DefaultTimeoutClientInterceptor(
                properties.getTimeout()
        ));
        interceptors.add(new TracerClientInterceptor(
                tracer
        ));
        var tvmHeaderClientHelper = getTvmHelper(tvm, properties);
        if (tvmHeaderClientHelper != null) {
            interceptors.add(tvmHeaderClientHelper.getInterceptor(properties.getTvm().getDestinationAlias()));
        }
        Channel channel = ManagedChannelBuilder
                .forTarget(properties.getTarget())
                .intercept(interceptors)
                .usePlaintext()
                .build();

        return BanditApiServiceV1Grpc.newBlockingStub(channel);
    }

    private TvmHeaderClientHelper getTvmHelper(TvmWrapper tvm, TrainBanditProperties properties) {
        if (properties.getTvm() != null && properties.getTvm().isEnabled() && tvm != null) {
            return new TvmHeaderClientHelper(tvm);
        } else {
            return null;
        }
    }
}
