package ru.yandex.travel.orders.services.train.tariffinfo;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;
import org.apache.lucene.document.Document;
import org.apache.lucene.document.Field;
import org.apache.lucene.document.StringField;
import org.apache.lucene.index.Term;
import org.apache.lucene.search.Query;
import org.apache.lucene.search.TermQuery;
import org.apache.lucene.search.TopDocs;
import org.jetbrains.annotations.NotNull;

import ru.yandex.travel.dicts.rasp.proto.TTrainTariffInfo;
import ru.yandex.travel.orders.configurations.TrainTariffInfoDataProviderProperties;
import ru.yandex.travel.yt_lucene_index.LuceneIndexBuilder;
import ru.yandex.travel.yt_lucene_index.LuceneIndexService;

@Slf4j
public class TrainTariffInfoService extends LuceneIndexService<TTrainTariffInfo> implements TrainTariffInfoDataProvider {
    private static String FIELD_IM_REQUEST_CODE = "ireqc";
    private static String FIELD_IM_RESPONSE_CODE = "irspc";
    private static String FIELD_CODE = "c";
    private static String FIELD_TITLE_RU = "tru";

    public TrainTariffInfoService(TrainTariffInfoDataProviderProperties params, LuceneIndexBuilder<TTrainTariffInfo> luceneIndexBuilder) {
        super(params, luceneIndexBuilder, "TrainTariffInfo");
    }

    @Override
    public String getImRequestCode(String tariffCode) {
        return getLuceneIndex().search(searcher -> {
            Query query = new TermQuery(new Term(FIELD_CODE, tariffCode));
            TopDocs topDocs = searcher.search(query, 1);

            Preconditions.checkState(topDocs.totalHits > 0, "No such IM Request Code for Tariff Code: %s ", tariffCode);

            return searcher.doc(topDocs.scoreDocs[0].doc)
                    .getField(FIELD_IM_REQUEST_CODE).stringValue();
        });
    }

    @Override
    // TODO (syukhno): remove everything below after transition to Tanker
    public String getOptionalTariffCode(String imResponseCode) {
        return getLuceneIndex().search(searcher -> {
            Query query = new TermQuery(new Term(FIELD_IM_RESPONSE_CODE, imResponseCode));
            TopDocs topDocs = searcher.search(query, 1);

            if (topDocs.totalHits == 0) {
                return null;
            }

            return searcher.doc(topDocs.scoreDocs[0].doc)
                    .getField(FIELD_CODE).stringValue();
        });
    }

    @Override
    public String getOptionalTariffTitle(String tariffCode) {
        if (tariffCode == null) {
            return null;
        }

        return getLuceneIndex().search(searcher -> {
            Query query = new TermQuery(new Term(FIELD_CODE, tariffCode));
            TopDocs topDocs = searcher.search(query, 1);

            if (topDocs.totalHits == 0) {
                return null;
            }

            return searcher.doc(topDocs.scoreDocs[0].doc)
                    .getField(FIELD_TITLE_RU).stringValue();
        });
    }

    @NotNull
    @Override
    protected Iterable<Document> documentProducer(TTrainTariffInfo row) {
        List<Document> documents = new ArrayList<>();
        Set<String> imResponseCodes = new HashSet<>();

        Arrays.stream(row.getImResponseCodes().split(",")).forEach(imResponseCode -> {
            Preconditions.checkState(!imResponseCodes.contains(imResponseCode),
                    "Tariff info has duplicate code %s", imResponseCode);

            imResponseCodes.add(imResponseCode);
            Document document = new Document();

            document.add(new StringField(FIELD_CODE, row.getCode(), Field.Store.YES));
            document.add(new StringField(FIELD_IM_REQUEST_CODE, row.getImRequestCode(), Field.Store.YES));
            document.add(new StringField(FIELD_TITLE_RU, row.getTitleRu(), Field.Store.YES));
            document.add(new StringField(FIELD_IM_RESPONSE_CODE, imResponseCode, Field.Store.YES));

            documents.add(document);
        });

        return documents;
    }
}
