package ru.yandex.travel.orders.workflows.invoice.aeroflot.handlers;

import java.text.DecimalFormat;
import java.text.DecimalFormatSymbols;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.AeroflotInvoice;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.entities.FiscalItemType;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustHotelsProperties;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketDeveloperPayload;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketResponse;
import ru.yandex.travel.orders.services.payments.model.TrustStartPaymentResponse;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.EAeroflotInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentCreated;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflows.order.aeroflot.AeroflotWorkflowUtils;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.configuration.AeroflotWorkflowProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

import static ru.yandex.travel.orders.workflows.invoice.aeroflot.handlers.TrustUtils.trustUserInfo;

@Service
@Slf4j
public class AeroflotInvoiceCreateTrustBasketStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<EAeroflotInvoiceState, AeroflotInvoice> {
    private final TrustHotelsProperties trustConfig;
    private final TrustClientProvider trustClientProvider;
    private final AeroflotWorkflowProperties aeroflotWorkflowProperties;
    private final ObjectMapper objectMapper = new ObjectMapper();


    public AeroflotInvoiceCreateTrustBasketStateHandler(
            TrustHotelsProperties trustConfig, TrustClientProvider trustClientProvider,
            AeroflotWorkflowProperties aeroflotWorkflowProperties
    ) {
        this.trustConfig = trustConfig;
        this.trustClientProvider = trustClientProvider;
        this.aeroflotWorkflowProperties = aeroflotWorkflowProperties;
    }

    @HandleEvent
    public void onCreateBasket(TPaymentCreated event,
                               StateContext<EAeroflotInvoiceState, AeroflotInvoice> stateContext) {
        AeroflotInvoice invoice = stateContext.getWorkflowEntity();
        AeroflotOrderItem orderItem = AeroflotWorkflowUtils.getOnlyOrderItem(invoice.getOrder());
        Money totalAmount = orderItem.getPayload().getActualCosts().getTotalAmount();

        TrustCreateBasketRequest request = TrustCreateBasketRequest.builder()
                .productId(FiscalItemType.FLIGHT_AEROFLOT.getTrustId())
                .amount(formatMoney(totalAmount))
                .currency(totalAmount.getCurrency().getCurrencyCode())
                .returnPath(invoice.getReturnPath())
                .templateTag(invoice.getSource())
                .isExternalBinding("1")
                .userEmail(invoice.getClientEmail())
                .userPhone(formatPhone(invoice.getClientPhone()))
                .paymentTimeout((double) aeroflotWorkflowProperties.getInvoiceTrustTimeout().getSeconds())
                .build();

        setDeveloperPayloadIfNeeded(invoice, request);
        TrustCreateBasketResponse createRsp =
                trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile())
                        .createBasket(request, trustUserInfo(invoice), invoice.getEffectivePaymentTestContext());
        invoice.setPurchaseToken(createRsp.getPurchaseToken());

        TrustStartPaymentResponse startResponse =
                trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile())
                        .startPayment(invoice.getPurchaseToken(), trustUserInfo(invoice));
        invoice.setPaymentUrl(startResponse.getPaymentUrl());

        stateContext.setState(EAeroflotInvoiceState.IS_WAIT_TRUST_TOKENIZATION);
        stateContext.scheduleEvent(TPaymentStarted.newBuilder().build());
        // for some future needs
        stateContext.scheduleExternalEvent(invoice.getOrderWorkflowId(), TInvoicePaymentStarted.newBuilder().build());
    }

    private void setDeveloperPayloadIfNeeded(Invoice invoice, TrustCreateBasketRequest request) {
        TrustCreateBasketDeveloperPayload developerPayload = new TrustCreateBasketDeveloperPayload();
        Order order = invoice.getOrder();
        EDisplayOrderType orderType = order != null ? order.getDisplayType() : null;
        String paymentFormTemplateName = trustConfig.getSettings(orderType).getPaymentWebFormTemplateName();
        developerPayload.setPaymentCompletionAction("spin");
        if (invoice.useNewCommonPaymentWebForm()) {
            paymentFormTemplateName = trustConfig.getSettings(orderType).getCommonPaymentWebForm();
            developerPayload.setPaymentCompletionAction("redirect");
            developerPayload.setEnablePaymentCancellation(true);
        }
        developerPayload.setTemplate(Strings.emptyToNull(paymentFormTemplateName));

        try {
            String developerPayloadStr = objectMapper.writeValueAsString(developerPayload);
            request.setDeveloperPayload(developerPayloadStr);
        } catch (JsonProcessingException e) {
            log.error("Couldn't serialize trust developer payload: " + developerPayload, e);
        }
    }

    private String formatPhone(String phone) {
        if (phone == null) {
            return null;
        }
        // todo(tlg-13): move the phone normalization code into a separate common helper
        String cleanedPhone = phone.replaceAll("[^\\d]", "");
        return cleanedPhone.substring(0, Math.min(15, cleanedPhone.length()));
    }

    private static String formatMoney(Money price) {
        DecimalFormat df = new DecimalFormat("#.##");
        DecimalFormatSymbols symbols = df.getDecimalFormatSymbols();
        symbols.setDecimalSeparator('.');
        df.setDecimalFormatSymbols(symbols);
        return df.format(price.getNumber().doubleValueExact());
    }
}
