package ru.yandex.travel.orders.workflows.invoice.aeroflot.handlers;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.AeroflotInvoice;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.EAeroflotInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoiceConfirmed;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoiceExpired;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoicePaymentFailed;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderPaymentConfirmationFailed;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderPaymentConfirmed;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Service
@RequiredArgsConstructor
@Slf4j
public class AeroflotInvoiceWaitConfirmationStateHandler extends AnnotatedStatefulWorkflowEventHandler<EAeroflotInvoiceState, AeroflotInvoice> {

    @HandleEvent
    public void handleInvoiceConfirmed(TAeroflotInvoiceConfirmed event,
                                       StateContext<EAeroflotInvoiceState, AeroflotInvoice> ctx) {
        Preconditions.checkArgument(!Strings.isNullOrEmpty(event.getPnr()), "Confirmed order doesn't have PNR");
        AeroflotInvoice invoice = ctx.getWorkflowEntity();
        ctx.setState(EAeroflotInvoiceState.IS_CONFIRMED);
        ctx.scheduleExternalEvent(invoice.getOrderWorkflowId(),
                TAeroflotOrderPaymentConfirmed.newBuilder()
                        .setPnr(event.getPnr())
                        .build());
    }

    @HandleEvent
    public void handlePaymentFailed(TAeroflotInvoicePaymentFailed event,
                                    StateContext<EAeroflotInvoiceState, AeroflotInvoice> ctx) {
        AeroflotInvoice invoice = ctx.getWorkflowEntity();
        ctx.setState(EAeroflotInvoiceState.IS_CANCELLED);
        ctx.scheduleExternalEvent(invoice.getOrderWorkflowId(),
                TAeroflotOrderPaymentConfirmationFailed.newBuilder()
                        .setFailed(true)
                        .build());
    }

    @HandleEvent
    public void handleInvoiceExpired(TAeroflotInvoiceExpired event,
                                     StateContext<EAeroflotInvoiceState, AeroflotInvoice> ctx) {

        AeroflotInvoice invoice = ctx.getWorkflowEntity();
        ctx.setState(EAeroflotInvoiceState.IS_TIMED_OUT);
        ctx.scheduleExternalEvent(invoice.getOrderWorkflowId(),
                TAeroflotOrderPaymentConfirmationFailed.newBuilder()
                        .setTimedOut(true)
                        .build());
    }
}
