package ru.yandex.travel.orders.workflows.invoice.aeroflot.handlers;

import java.time.Instant;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.AeroflotInvoice;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.EAeroflotInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoiceAutoConfirmation;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoiceOrderCreationFailed;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoicePaymentConfirmation;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotInvoiceAutoConfirmed;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotInvoiceCancelled;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.configuration.AeroflotWorkflowProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Service
@RequiredArgsConstructor
@Slf4j
public class AeroflotInvoiceWaitOrderCreatedStateHandler extends AnnotatedStatefulWorkflowEventHandler<EAeroflotInvoiceState, AeroflotInvoice> {
    private final AeroflotWorkflowProperties properties;

    @HandleEvent
    public void handleAeroflotInvoicePaymentConfirmation(TAeroflotInvoicePaymentConfirmation event,
                                                         StateContext<EAeroflotInvoiceState, AeroflotInvoice> ctx) {
        AeroflotInvoice invoice = ctx.getWorkflowEntity();
        Instant newExpiresAt = Instant.now().plus(properties.getInvoiceConfirmationTimeout());
        log.info("The order has been created, starting confirmation checks; expires at {}", newExpiresAt);

        invoice.setExpirationDate(newExpiresAt);
        invoice.setConfirmationUrl(event.getConfirmationUrl());
        ctx.setState(EAeroflotInvoiceState.IS_WAIT_CONFIRMATION);

        ctx.getWorkflowEntity().setNextCheckStatusAt(Instant.now());
        ctx.getWorkflowEntity().setBackgroundJobActive(true);
    }

    @HandleEvent
    public void onOrderCreationFailed(TAeroflotInvoiceOrderCreationFailed event, StateContext<EAeroflotInvoiceState,
            AeroflotInvoice> ctx) {
        ctx.setState(EAeroflotInvoiceState.IS_CANCELLED);
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(),
                TAeroflotInvoiceCancelled.newBuilder().build());
    }

    @HandleEvent
    public void onAutoConfirmed(TAeroflotInvoiceAutoConfirmation event, StateContext<EAeroflotInvoiceState,
            AeroflotInvoice> ctx) {
        ctx.setState(EAeroflotInvoiceState.IS_CONFIRMED);
        ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(),
                TAeroflotInvoiceAutoConfirmed.newBuilder().build());
    }
}
