package ru.yandex.travel.orders.workflows.invoice.aeroflot.handlers;

import java.time.Instant;
import java.util.Set;

import com.google.common.collect.ImmutableSet;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.AeroflotInvoice;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.EAeroflotInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoiceCardTokenizationFailed;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoiceCardTokenized;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoiceExpired;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentStarted;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderCardTokenizationFailed;
import ru.yandex.travel.orders.workflow.order.aeroflot.proto.TAeroflotOrderCardTokenized;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;


@Service
@RequiredArgsConstructor
@Slf4j
public class AeroflotInvoiceWaitTrustTokenizationStateHandler extends AnnotatedStatefulWorkflowEventHandler<EAeroflotInvoiceState, AeroflotInvoice> {
    private static final Set<String> EXPECTED_TOKENIZATION_ERRORS = ImmutableSet.of(
            // more Trust error codes: https://wiki.yandex-team.ru/Trust/Payments/RC/
            "user_cancelled", // user has closed payment form
            "payment_timeout", // no form submission happened in time
            "invalid_xrf_token", // user cookie problems
            "restricted_card" // blocked cards
    );
    private static final String TOKENIZATION_UNKNOWN_ERROR_CODE = "unknown_error";
    private static final Set<String> EXPECTED_TOKENIZATION_UNKNOWN_ERRORS = Set.of(
            "invalid_expiration_date" // TRUSTDUTY-2474
    );

    private final StarTrekService starTrekService;

    @HandleEvent
    public void onEvent(TPaymentStarted event, StateContext<EAeroflotInvoiceState, AeroflotInvoice> ctx) {
        AeroflotInvoice invoice = ctx.getWorkflowEntity();
        log.info("Payment has started, starting status polling");
        ctx.getWorkflowEntity().setNextCheckStatusAt(Instant.now());
        ctx.getWorkflowEntity().setBackgroundJobActive(true);
    }

    @HandleEvent
    public void handleTokenizationFailed(TAeroflotInvoiceCardTokenizationFailed event,
                                         StateContext<EAeroflotInvoiceState, AeroflotInvoice> ctx) {

        AeroflotInvoice invoice = ctx.getWorkflowEntity();
        log.info("Card tokenization has failed; authErrorCode={}", invoice.getAuthorizationErrorCode());

        var basketStatus = ProtoUtils.fromTJson(event.getBasketStatus(), TrustBasketStatusResponse.class);
        if (basketStatus != null) {
            invoice.setAuthorizationErrorCode(basketStatus.getPaymentRespCode());
            invoice.setAuthorizationErrorDesc(basketStatus.getPaymentRespDesc());

            invoice.setTrustPaymentId(basketStatus.getTrustPaymentId());
            invoice.setUserAccount(basketStatus.getUserAccount());
            if (basketStatus.getStartTs() != null) {
                invoice.setPaymentStartTs(basketStatus.getStartTs());
            }
        }
        if (basketStatus == null || !isExpectedTrustError(basketStatus)) {
            String failureDetails = basketStatus != null ? basketStatus.toString() : "No data available";
            starTrekService.createIssueForAeroflotFailedTokenization(
                    invoice.getOrder(), invoice.getPurchaseToken(), failureDetails, ctx);
        }

        ctx.setState(EAeroflotInvoiceState.IS_CANCELLED);
        ctx.scheduleExternalEvent(invoice.getOrderWorkflowId(),
                TAeroflotOrderCardTokenizationFailed.newBuilder().build());
    }

    @SuppressWarnings("RedundantIfStatement")
    private boolean isExpectedTrustError(TrustBasketStatusResponse status) {
        if (EXPECTED_TOKENIZATION_ERRORS.contains(status.getPaymentRespCode())) {
            return true;
        }
        if (TOKENIZATION_UNKNOWN_ERROR_CODE.equals(status.getPaymentRespCode())
                && EXPECTED_TOKENIZATION_UNKNOWN_ERRORS.contains(status.getPaymentRespDesc())) {
            return true;
        }
        return false;
    }

    @HandleEvent
    public void handleCardTokenized(TAeroflotInvoiceCardTokenized event,
                                    StateContext<EAeroflotInvoiceState, AeroflotInvoice> ctx) {
        AeroflotInvoice invoice = ctx.getWorkflowEntity();
        ctx.setState(EAeroflotInvoiceState.IS_WAIT_ORDER_CREATED);
        ctx.scheduleExternalEvent(invoice.getOrderWorkflowId(),
                TAeroflotOrderCardTokenized.newBuilder().setTokenizedCard(event.getTokenizedCard()).build());

        var basketStatus = ProtoUtils.fromTJson(event.getBasketStatus(), TrustBasketStatusResponse.class);
        if (basketStatus != null) {
            invoice.setTrustPaymentId(basketStatus.getTrustPaymentId());
            invoice.setUserAccount(basketStatus.getUserAccount());
            invoice.setApprovalCode(basketStatus.getApprovalCode());
            if (basketStatus.getStartTs() != null) {
                invoice.setPaymentStartTs(basketStatus.getStartTs());
            }
            if (basketStatus.getPaymentTs() != null) {
                invoice.setTrustPaymentTs(basketStatus.getPaymentTs());
            }
        }
    }

    @HandleEvent
    public void handleExpired(TAeroflotInvoiceExpired event,
                              StateContext<EAeroflotInvoiceState, AeroflotInvoice> ctx) {
        AeroflotInvoice invoice = ctx.getWorkflowEntity();
        log.info("Card tokenization has timed out; orderId={}", invoice.getOrder().getId());
        ctx.setState(EAeroflotInvoiceState.IS_TIMED_OUT);
        ctx.scheduleExternalEvent(invoice.getOrderWorkflowId(),
                TAeroflotOrderCardTokenizationFailed.newBuilder().build());
    }
}
