package ru.yandex.travel.orders.workflows.invoice.aeroflot.jobs;

import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.orders.workflows.orderitem.aeroflot.configuration.AeroflotWorkflowProperties;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorProperties;

@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties(AeroflotWorkflowProperties.class)
public class AeroflotInvoiceProcessorsConfiguration {
    private final AeroflotInvoiceRefreshService aeroflotInvoiceRefreshService;
    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;
    private final AeroflotWorkflowProperties config;

    @Bean
    public TaskProcessor<UUID> aeroflotInvoiceAwaitingTokenizationProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return aeroflotInvoiceRefreshService.fetchInvoicesWaitingTokenization(getLockedTaskKeys(),
                        maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return aeroflotInvoiceRefreshService.countInvoicesWaitingTokenization(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("AeroflotAwaitingTokenization");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, aeroflotInvoiceRefreshService::refreshInvoiceWaitingTokenization,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("AeroflotAwaitTokenization").daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(config.getMaxRefreshTokenization())
                        .initialStartDelay(config.getInvoiceAwaitingTokenizationRefreshInitialDelay())
                        .scheduleRate(config.getInvoiceAwaitingTokenizationRefreshRate())
                        .build());
    }

    @Bean
    public TaskProcessor<UUID> aeroflotInvoiceAwaitingConfirmationProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return aeroflotInvoiceRefreshService.fetchInvoicesWaitingConfirmation(getLockedTaskKeys(),
                        maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return aeroflotInvoiceRefreshService.countInvoicesWaitingConfirmation(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("AeroflotAwaitingConfirmation");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, aeroflotInvoiceRefreshService::refreshInvoiceWaitingConfirmation,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("AeroflotAwaitConfirmation").daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(config.getMaxRefreshConfirmation())
                        .initialStartDelay(config.getInvoiceAwaitingConfirmationRefreshInitialDelay())
                        .scheduleRate(config.getInvoiceAwaitingConfirmationRefreshRate())
                        .build());
    }
}
