package ru.yandex.travel.orders.workflows.invoice.aeroflot.jobs;

import java.time.Instant;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.AeroflotProviderProperties;
import ru.yandex.avia.booking.partners.gateways.aeroflot.demo.AeroflotDemoCardTokenizer;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderCreateResult;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderRef;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotOrderStatus;
import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.AeroflotInvoice;
import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.repository.AeroflotInvoiceRepository;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.model.PaymentStatusEnum;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustBindingToken;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.EAeroflotInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoiceCardTokenizationFailed;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoiceCardTokenized;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoiceConfirmed;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoiceExpired;
import ru.yandex.travel.orders.workflow.invoice.aeroflot.proto.TAeroflotInvoicePaymentFailed;
import ru.yandex.travel.orders.workflows.order.aeroflot.AeroflotWorkflowUtils;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.configuration.AeroflotWorkflowProperties;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.provider.AeroflotServiceProvider;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMessageSender;

import static ru.yandex.travel.orders.workflows.invoice.trust.handlers.TrustUserInfoHelper.createUserInfo;

@Service
@RequiredArgsConstructor
@Slf4j
@EnableConfigurationProperties(AeroflotWorkflowProperties.class)
public class AeroflotInvoiceRefreshService {
    private static final Set<PaymentStatusEnum> TERMINAL_TOKENIZATION_STATUSES =
            Set.of(PaymentStatusEnum.CLEARED, PaymentStatusEnum.CANCELED, PaymentStatusEnum.NOT_AUTHORIZED);

    private final AeroflotInvoiceRepository aeroflotInvoiceRepository;

    private final WorkflowMessageSender workflowMessageSender;

    private final AeroflotServiceProvider provider;

    private final AeroflotWorkflowProperties aeroflotWorkflowProperties;

    private final AeroflotProviderProperties providerProperties;

    private final AeroflotDemoCardTokenizer aeroflotDemoCardTokenizer;

    private final TrustClientProvider trustClientProvider;

    @TransactionMandatory
    public List<UUID> fetchInvoicesWaitingTokenization(Set<UUID> activeInvoices, int maxResultSize) {
        return aeroflotInvoiceRepository.getInvoiceIdsAwaitingRefreshInStateWithExclusions(Instant.now(),
                EAeroflotInvoiceState.IS_WAIT_TRUST_TOKENIZATION, EWorkflowState.WS_RUNNING,
                Collections.unmodifiableSet(activeInvoices), PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    public long countInvoicesWaitingTokenization(Set<UUID> activeInvoices) {
        return aeroflotInvoiceRepository.countInvoicesAwaitingRefreshInStateWithExclusions(Instant.now(),
                EAeroflotInvoiceState.IS_WAIT_TRUST_TOKENIZATION, EWorkflowState.WS_RUNNING, activeInvoices);
    }

    @TransactionMandatory
    public List<UUID> fetchInvoicesWaitingConfirmation(Set<UUID> activeInvoices, int maxResultSize) {
        return aeroflotInvoiceRepository.getInvoiceIdsAwaitingRefreshInStateWithExclusions(Instant.now(),
                EAeroflotInvoiceState.IS_WAIT_CONFIRMATION, EWorkflowState.WS_RUNNING,
                Collections.unmodifiableSet(activeInvoices), PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    public long countInvoicesWaitingConfirmation(Set<UUID> activeInvoices) {
        return aeroflotInvoiceRepository.countInvoicesAwaitingRefreshInStateWithExclusions(Instant.now(),
                EAeroflotInvoiceState.IS_WAIT_CONFIRMATION, EWorkflowState.WS_RUNNING, activeInvoices);
    }

    @TransactionMandatory
    public void refreshInvoiceWaitingTokenization(UUID invoiceId) {
        AeroflotInvoice invoice = aeroflotInvoiceRepository.getOne(invoiceId);
        try (var ignored2 = NestedMdc.forEntity(invoice)) {
            if (!invoice.isBackgroundJobActive()) {
                log.info("Background job not active, returning");
                return;
            }

            TrustBasketStatusResponse statusRsp =
                    trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile())
                            .getBasketStatus(invoice.getPurchaseToken(), createUserInfo(invoice));
            log.debug("Tokenization basket: basket_id={}, status={}", invoice.getPurchaseToken(),
                    statusRsp.getPaymentStatus());
            if (
                    "error".equals(statusRsp.getPaymentRespCode()) // RASPTICKETS-22636
                            && "shutdown in progress".equals(statusRsp.getPaymentRespDesc())
            ) {
                log.info("Reschedule reason: shutdown in progress");
                invoice.rescheduleNextCheckStatusAt(aeroflotWorkflowProperties.getInvoiceTrustRefreshTimeout());
                return;
            }
            if (TERMINAL_TOKENIZATION_STATUSES.contains(statusRsp.getPaymentStatus())) {
                if (statusRsp.getPaymentStatus() != PaymentStatusEnum.CLEARED) {
                    log.info("Tokenization failed: purchaseToken={}; paymentStatus={}, respCode={}, respDesc={}",
                            invoice.getPurchaseToken(), statusRsp.getPaymentStatus(),
                            statusRsp.getPaymentRespCode(), statusRsp.getPaymentRespDesc());
                    workflowMessageSender.scheduleEvent(invoice.getWorkflow().getId(),
                            TAeroflotInvoiceCardTokenizationFailed.newBuilder()
                                    .setBasketStatus(ProtoUtils.toTJson(statusRsp))
                                    .build());
                    invoice.setNextCheckStatusAt(null);
                    invoice.setBackgroundJobActive(false);
                    return;
                }
                TrustBindingToken token = statusRsp.getBindingToken();
                Preconditions.checkNotNull(token, "No binding token: statusRsp=%s", statusRsp);
                log.info("Tokenized card: expiration={}, token={}-<hidden>", token.getExpiration(),
                        token.getValue().substring(0, 8));
                String tokenizedCard;
                // the <no_3ds> card doesn't work in test Trust environment
                if (providerProperties.isEnableTestingScenarios() && invoice.getClientEmail().contains(
                        "+demo_no_3ds")) {
                    log.warn("Running demo <no_3ds> card tokenization; invoice_id={}, order_id={}",
                            invoice.getId(),
                            invoice.getOrder().getId());
                    tokenizedCard = aeroflotDemoCardTokenizer.tokenizeDemoCardNo3ds();
                } else {
                    log.debug("Using Trust-tokenized card");
                    tokenizedCard = token.getValue();
                }
                workflowMessageSender.scheduleEvent(invoice.getWorkflow().getId(),
                        TAeroflotInvoiceCardTokenized.newBuilder()
                                .setBasketStatus(ProtoUtils.toTJson(statusRsp))
                                .setTokenizedCard(tokenizedCard)
                                .build());
                invoice.setNextCheckStatusAt(null);
                invoice.setBackgroundJobActive(false);
            } else if (invoice.getExpirationDate().isBefore(Instant.now())) {
                log.info("The payment has timed out, cancelling it; invoice_id={}", invoice.getId());
                workflowMessageSender.scheduleEvent(invoice.getWorkflow().getId(),
                        TAeroflotInvoiceExpired.newBuilder().build());
                invoice.setNextCheckStatusAt(null);
                invoice.setBackgroundJobActive(false);
            } else {
                invoice.rescheduleNextCheckStatusAt(aeroflotWorkflowProperties.getInvoiceTrustRefreshTimeout());
            }
        }
    }

    @TransactionMandatory
    public void refreshInvoiceWaitingConfirmation(UUID invoiceId) {
        AeroflotInvoice invoice = aeroflotInvoiceRepository.getOne(invoiceId);
        try (var ignored = NestedMdc.forEntity(invoice.getId(), invoice.getEntityType())) {

            if (!invoice.isBackgroundJobActive()) {
                log.info("Background job not active, returning");
                return;
            }
            AeroflotOrder order = invoice.getOrder();
            AeroflotOrderItem service = AeroflotWorkflowUtils.getOnlyOrderItem(invoice.getOrder());
            AeroflotOrderCreateResult result;
            try {
                result = provider.getAeroflotServiceForProfile(service)
                        .getOrderStatus(order.getId(), service.getPayload());
            } catch (Exception e) {
                log.info("Aeroflot order payment confirmation check failed", e);
                // there could be temporary errors, we will retry until the expiration happens
                result = AeroflotOrderCreateResult.builder()
                        .statusCode(AeroflotOrderStatus.PAYMENT_PARTIALLY_FAILED)
                        .orderRef(AeroflotOrderRef.builder()
                                .orderId(service.getPayload().getBookingRef().getOrderId())
                                .build())
                        .build();
            }
            boolean expired = invoice.getExpirationDate().isBefore(Instant.now())
                    || result.isExpired();
            if (result.isPaid() && (result.getOrderRef().getPnr() != null || expired)) {
                // no pnr payment should be failed by the workflow,
                // not throwing exceptions here to keep the task processor running
                log.info("Payment has been completed; pnr {}; expired - {}; expires at {}; status {}",
                        result.getOrderRef().getPnr(), expired, invoice.getExpirationDate(), result.getStatusCode());
                invoice.setBackgroundJobActive(false);
                workflowMessageSender.scheduleEvent(invoice.getWorkflow().getId(),
                        TAeroflotInvoiceConfirmed.newBuilder()
                                .setPnr(Strings.nullToEmpty(result.getOrderRef().getPnr()))
                                .build());
                // the logic of this status has changed recently
                // it's not recommended to call the status method until 3ds confirmation is done by the user
                // but since we don't want to rely on browser redirect signals we still continue polling it
            } else if (result.getStatusCode() == AeroflotOrderStatus.PAYMENT_FAILED && expired) {
                log.info("The payment has failed, cancelling the invoice");
                invoice.setBackgroundJobActive(false);
                workflowMessageSender.scheduleEvent(invoice.getWorkflow().getId(),
                        TAeroflotInvoicePaymentFailed.newBuilder().build());
            } else if (expired) {
                log.info("The payment has timed out, cancelling it");
                invoice.setBackgroundJobActive(false);
                workflowMessageSender.scheduleEvent(invoice.getWorkflow().getId(),
                        TAeroflotInvoiceExpired.newBuilder().build());
            } else {
                if (result.isPaid()) {
                    log.info("The order is paid but no pnr is available at the moment, will wait more");
                }
                invoice.rescheduleNextCheckStatusAt(aeroflotWorkflowProperties.getInvoiceConfirmationRefreshTimeout());
            }
        }
    }
}
