package ru.yandex.travel.orders.workflows.invoice.refund;

import java.time.LocalDateTime;
import java.util.Collection;
import java.util.List;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.repository.SimpleTrustRefundRepository;
import ru.yandex.travel.orders.workflow.trust.refund.proto.TRefreshStatus;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowProcessService;

@Service
@Slf4j
@RequiredArgsConstructor
public class TrustRefundRefreshService {

    private final SimpleTrustRefundRepository simpleTrustRefundRepository;
    private final WorkflowProcessService processService;

    @TransactionMandatory
    public List<UUID> getRefundsWaitingForRefresh(Collection<UUID> excludedIds, int maxResultSize) {
        return simpleTrustRefundRepository.getTrustRefundWaitingForRefreshStatus(
                excludedIds,
                LocalDateTime.now(),
                EWorkflowState.WS_RUNNING,
                PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    public Long countAllRefundsWaitingForRefresh(Collection<UUID> excludedIds) {
        return simpleTrustRefundRepository.countTrustRefundWaitingForRefreshStatus(
                excludedIds,
                LocalDateTime.now(),
                EWorkflowState.WS_RUNNING);
    }

    @TransactionMandatory
    public void trustRefundStatusRefresh(UUID refundId) {
        var trustRefund = simpleTrustRefundRepository.getOne(refundId);
        try (var ignored = NestedMdc.forEntity(trustRefund.getId(), trustRefund.getEntityType())) {
            trustRefund.setSystemRefreshScheduled(true);
            processService.scheduleEvent(trustRefund.getWorkflow().getId(), TRefreshStatus.newBuilder().build());
        }
    }
}
