package ru.yandex.travel.orders.workflows.invoice.refund.handlers;

import java.util.HashMap;
import java.util.Map;

import com.google.common.base.Strings;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.entities.SimpleTrustRefund;
import ru.yandex.travel.orders.entities.TrustRefundItem;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.model.TrustCompositeOrderPaymentMarkup;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateRefundResponse;
import ru.yandex.travel.orders.workflow.trust.refund.proto.ETrustRefundState;
import ru.yandex.travel.orders.workflow.trust.refund.proto.TCreateRefund;
import ru.yandex.travel.orders.workflow.trust.refund.proto.TStartRefund;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.orders.workflows.invoice.trust.handlers.TrustUserInfoHelper;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Service
@Slf4j
public class CreateStateHandler extends AnnotatedStatefulWorkflowEventHandler<ETrustRefundState, SimpleTrustRefund> {

    private final TrustClientProvider trustClientProvider;

    public CreateStateHandler(TrustClientProvider trustClientProvider) {
        this.trustClientProvider = trustClientProvider;
    }

    @HandleEvent
    public void handleCreateRefund(TCreateRefund message,
                                   StateContext<ETrustRefundState, SimpleTrustRefund> ctx) {

        Invoice invoice = ctx.getWorkflowEntity().getInvoice();
        TrustUserInfo trustUserInfo = TrustUserInfoHelper.createUserInfo(invoice);
        TrustClient trustClient = trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile());

        TrustCreateRefundResponse rsp =
                trustClient.createRefund(fromTrustRefund(ctx.getWorkflowEntity()), trustUserInfo);

        ctx.getWorkflowEntity().setTrustRefundId(rsp.getTrustRefundId());

        ctx.setState(ETrustRefundState.RS_START);
        ctx.scheduleEvent(TStartRefund.newBuilder().build());
    }

    private TrustCreateRefundRequest fromTrustRefund(SimpleTrustRefund refund) {
        TrustCreateRefundRequest result = new TrustCreateRefundRequest();
        result.setPurchaseToken(refund.getInvoice().getPurchaseToken());
        result.setReasonDesc(refund.getDescription());
        if (Strings.isNullOrEmpty(result.getReasonDesc())) {
            log.warn("Empty refund reason description, using the default fallback value; trust refund {}",
                    refund.getId(), new IllegalStateException("Empty reason_desc"));
            result.setReasonDesc("Refund - other");
        }
        Map<String, MoneyMarkup> moneyMarkups = new HashMap<>();
        boolean isCardOnly = refund.getInvoice().getInvoiceItems().stream()
                .allMatch(ii -> ii.getPriceMarkup().isCardOnly());
        for (TrustRefundItem item : refund.getRefundItems()) {
            result.addOrder(item.getTrustOrderId(), item.getRefundDelta());
            if (!isCardOnly) {
                MoneyMarkup markup = item.getRefundDeltaMarkup();
                if (moneyMarkups.get(item.getTrustOrderId()) != null) {
                    moneyMarkups.put(item.getTrustOrderId(), moneyMarkups.get(item.getTrustOrderId()).add(markup));
                } else {
                    moneyMarkups.put(item.getTrustOrderId(), markup);
                }
            }
        }
        if (!moneyMarkups.isEmpty()) {
            Map<String, TrustCompositeOrderPaymentMarkup> refundsMarkup = new HashMap<>();
            moneyMarkups.forEach((id, markup) -> {
                TrustCompositeOrderPaymentMarkup trustMarkup = InvoiceUtils.toTrustPaymentMarkup(markup);
                // zero values cause refund failure
                trustMarkup = trustMarkup.withoutZeros();
                refundsMarkup.put(id, trustMarkup);
            });
            result.setPaymethodMarkup(refundsMarkup);
        }
        return result;
    }
}
