package ru.yandex.travel.orders.workflows.invoice.refund.handlers;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.SimpleTrustRefund;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.model.TrustRefundStatusResponse;
import ru.yandex.travel.orders.workflow.invoice.proto.TRefundError;
import ru.yandex.travel.orders.workflow.invoice.proto.TRefundFailed;
import ru.yandex.travel.orders.workflow.invoice.proto.TRefundSuccess;
import ru.yandex.travel.orders.workflow.trust.refund.proto.ETrustRefundState;
import ru.yandex.travel.orders.workflow.trust.refund.proto.TRefreshStatus;
import ru.yandex.travel.orders.workflows.invoice.trust.handlers.TrustUserInfoHelper;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Service
@Slf4j
public class InProcessStateHandler extends AnnotatedStatefulWorkflowEventHandler<ETrustRefundState, SimpleTrustRefund> {
    private final TrustClientProvider trustClientProvider;

    public InProcessStateHandler(TrustClientProvider trustClientProvider) {
        this.trustClientProvider = trustClientProvider;
    }

    @HandleEvent
    public void handleRefreshStatus(TRefreshStatus event,
                                    StateContext<ETrustRefundState, SimpleTrustRefund> ctx) {
        SimpleTrustRefund refund = ctx.getWorkflowEntity();
        refund.setSystemRefreshScheduled(false);
        Invoice invoice = ctx.getWorkflowEntity().getInvoice();
        TrustUserInfo trustUserInfo = TrustUserInfoHelper.createUserInfo(invoice);
        TrustRefundStatusResponse rsp = trustClientProvider
                .getTrustClientForPaymentProfile(invoice.getPaymentProfile())
                .getRefundStatus(ctx.getWorkflowEntity().getTrustRefundId(), trustUserInfo);
        switch (rsp.getStatus()) {
            case SUCCESS:
                refund.setNextCheckStatusAt(null);
                ctx.setState(ETrustRefundState.RS_SUCCESS);
                ctx.scheduleExternalEvent(
                        invoice.getWorkflow().getId(),
                        TRefundSuccess.newBuilder().build()
                );
                break;
            case FAILED:
                refund.setNextCheckStatusAt(null);
                ctx.setState(ETrustRefundState.RS_FAILED);
                ctx.scheduleExternalEvent(
                        invoice.getWorkflow().getId(),
                        TRefundFailed.newBuilder().build()
                );
                break;
            case ERROR:
                refund.setNextCheckStatusAt(null);
                ctx.setState(ETrustRefundState.RS_ERROR);
                ctx.scheduleExternalEvent(
                        invoice.getWorkflow().getId(),
                        TRefundError.newBuilder().build()
                );
                break;
            case WAIT_FOR_NOTIFICATION:
                refund.rescheduleNextRefreshAt();
                log.info("Trust status for refund {} has not changed", ctx.getWorkflowEntity().getId());
                break;
        }
    }
}
