package ru.yandex.travel.orders.workflows.invoice.refund.jobs;

import java.time.Duration;
import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.orders.services.payments.TrustHotelsProperties;
import ru.yandex.travel.orders.workflows.invoice.refund.TrustRefundRefreshService;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorProperties;

@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties(TrustHotelsProperties.class)
public class RefundTaskProcessorConfiguration {

    private final TrustRefundRefreshService refundRoutines;
    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;
    private final TrustHotelsProperties config;

    @Bean
    public TaskProcessor<UUID> trustRefundsWaitingRefreshProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return refundRoutines.getRefundsWaitingForRefresh(getLockedTaskKeys(), maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return refundRoutines.countAllRefundsWaitingForRefresh(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("TrustRefundsRefresh");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, refundRoutines::trustRefundStatusRefresh,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("TrustRefundsRefresh").daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(config.getNumberOfRefundRefreshThreads())
                        .initialStartDelay(Duration.ofMillis(50)).scheduleRate(Duration.ofSeconds(1)).build());
    }
}
