package ru.yandex.travel.orders.workflows.invoice.trust;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.util.List;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import org.javamoney.moneta.Money;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.InvoiceItem;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponseOrder;
import ru.yandex.travel.orders.services.payments.model.TrustCompositeOrderPaymentMarkup;
import ru.yandex.travel.orders.workflow.invoice.proto.TMoneyMarkup;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentRefund;
import ru.yandex.travel.orders.workflows.orderitem.RefundingUtils;

public class InvoiceUtils {
    private InvoiceUtils() {
    }

    public static void checkSums(Invoice invoice, TrustBasketStatusResponse trustBasketStatusResponse) {
        Map<String, InvoiceItem> trustOrderToInvoiceItem = invoice.getInvoiceItems().stream().collect(
                Collectors.toMap(InvoiceItem::getTrustOrderId, Function.identity())
        );

        Map<String, BigDecimal> currentOrderSums = trustBasketStatusResponse.getOrders().stream()
                .collect(Collectors.toMap(TrustBasketStatusResponseOrder::getOrderId, TrustBasketStatusResponseOrder::getPaidAmount));

        for (String invoiceItemTrustOrderId : trustOrderToInvoiceItem.keySet()) {
            if (!currentOrderSums.containsKey(invoiceItemTrustOrderId)) {
                throw new RuntimeException(
                        MessageFormat.format("Invoice item with trust order id {0} found, and no corresponding order id found in payment status", invoiceItemTrustOrderId)
                );
            }
        }

        for (Map.Entry<String, BigDecimal> trustOrderEntry : currentOrderSums.entrySet()) {
            InvoiceItem item = trustOrderToInvoiceItem.get(trustOrderEntry.getKey());
            if (item == null) {
                throw new RuntimeException(MessageFormat.format("Invoice item with trust order id {0} not found", trustOrderEntry.getKey()));
            }
            if (item.getPrice().compareTo(trustOrderEntry.getValue()) != 0) {
                throw new RuntimeException(
                        MessageFormat.format("Sums mismatch for invoice item {0}, item sum {1} and trust sum {2}", item.getId(), item.getPrice(), trustOrderEntry.getValue())
                );
            }
        }
    }

    public static TPaymentRefund buildFullRefund(Invoice invoice) {
        return buildFullRefund(invoice, "Full refund");
    }

    public static TPaymentRefund buildFullRefund(Invoice invoice, String reason) {
        Map<Long, Money> targetFiscalItems = buildFullRefundTargetFiscalItems(invoice.getInvoiceItems());
        Map<Long, MoneyMarkup> targetFiscalItemsMarkup =
                buildFullRefundTargetFiscalItemsMarkup(invoice.getInvoiceItems());
        return TPaymentRefund.newBuilder()
                .putAllTargetFiscalItems(RefundingUtils.convertTargetFiscalItemsToProto(targetFiscalItems))
                .putAllTargetFiscalItemsMarkup(RefundingUtils.convertTargetFiscalItemsMarkupToProto(targetFiscalItemsMarkup))
                .setReason(reason)
                .build();
    }

    public static Map<Long, Money> buildFullRefundTargetFiscalItems(List<InvoiceItem> invoiceItems) {
        return invoiceItems.stream()
                .collect(Collectors.toMap(InvoiceItem::getFiscalItemId, ii -> Money.zero(ii.getCurrency())));
    }

    public static Map<Long, MoneyMarkup> buildFullRefundTargetFiscalItemsMarkup(List<InvoiceItem> invoiceItems) {
        return invoiceItems.stream()
                .collect(Collectors.toMap(InvoiceItem::getFiscalItemId, ii -> MoneyMarkup.zero(ii.getCurrency())));
    }

    public static TrustCompositeOrderPaymentMarkup toTrustPaymentMarkup(MoneyMarkup markup) {
        markup.ensureValuesScale();
        return TrustCompositeOrderPaymentMarkup.builder()
                .card(markup.getCard().getNumberStripped())
                .yandexAccount(markup.getYandexAccount().getNumberStripped())
                .build();
    }

    public static TMoneyMarkup toProtoPaymentMarkup(MoneyMarkup markup) {
        return TMoneyMarkup.newBuilder()
                .setCard(ProtoUtils.toTPrice(markup.getCard()))
                .setYandexAccount(ProtoUtils.toTPrice(markup.getYandexAccount()))
                .build();
    }
}
