package ru.yandex.travel.orders.workflows.invoice.trust.handlers;

import java.text.MessageFormat;
import java.util.Map;
import java.util.function.Function;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.InvoiceItem;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.entities.SimpleTrustRefund;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.repository.SimpleTrustRefundRepository;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentClear;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentRefund;
import ru.yandex.travel.orders.workflow.invoice.proto.TScheduleClearing;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustInvoiceCallbackReceived;
import ru.yandex.travel.orders.workflow.trust.refund.proto.ETrustRefundState;
import ru.yandex.travel.orders.workflow.trust.refund.proto.TRefundCreated;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static ru.yandex.travel.orders.workflows.orderitem.RefundingUtils.convertTargetFiscalItemsFromProto;
import static ru.yandex.travel.orders.workflows.orderitem.RefundingUtils.convertTargetFiscalItemsMarkupFromProto;

@Service
@Slf4j
@RequiredArgsConstructor
@IgnoreEvents(types = {TPaymentClear.class, TScheduleClearing.class, TTrustInvoiceCallbackReceived.class})
public class ClearedStateHandler extends AnnotatedStatefulWorkflowEventHandler<ETrustInvoiceState, TrustInvoice> {

    private final SimpleTrustRefundRepository simpleTrustRefundRepository;

    private final WorkflowRepository workflowRepository;

    @HandleEvent
    public void handlePaymentRefund(TPaymentRefund event,
                                    StateContext<ETrustInvoiceState, TrustInvoice> stateContext) {
        //TODO (mbobrov) check for sum
        TrustInvoice invoice = stateContext.getWorkflowEntity();
        Map<Long, InvoiceItem> fiscalItemIds = invoice.getInvoiceItems().stream().collect(
                Collectors.toMap(InvoiceItem::getFiscalItemId, Function.identity())
        );

        // TODO (mbobrov) think of moving creation of workflow for entity to a separate service
        SimpleTrustRefund newRefund = new SimpleTrustRefund();
        newRefund.setDescription(event.getReason());
        newRefund.setState(ETrustRefundState.RS_NEW);
        newRefund.setInvoice(invoice);
        newRefund.setOrderRefundId(ProtoUtils.fromStringOrNull(event.getOrderRefundId()));
        invoice.setActiveTrustRefund(newRefund);

        Map<Long, Money> targetFiscalItems = convertTargetFiscalItemsFromProto(event.getTargetFiscalItemsMap());
        Map<Long, MoneyMarkup> targetFiscalItemsMarkup =
                convertTargetFiscalItemsMarkupFromProto(event.getTargetFiscalItemsMarkupMap());
        for (Map.Entry<Long, Money> refundEntry : targetFiscalItems.entrySet()) {
            InvoiceItem invoiceItem = fiscalItemIds.get(refundEntry.getKey());
            if (invoiceItem == null) {
                throw new RuntimeException(MessageFormat.format("Invoice item with fiscal id {0} not found",
                        refundEntry.getKey()));
            }
            MoneyMarkup currentItemsMarkup = invoiceItem.getPriceMarkup();
            MoneyMarkup targetItemsMarkup = targetFiscalItemsMarkup.get(refundEntry.getKey());
            newRefund.addRefundItem(
                    invoiceItem.getTrustOrderId(),
                    invoiceItem.getPrice(),
                    refundEntry.getValue().getNumberStripped(),
                    currentItemsMarkup,
                    targetItemsMarkup
            );
        }
        simpleTrustRefundRepository.save(newRefund);

        Workflow workflow = Workflow.createWorkflowForEntity(newRefund, invoice.getWorkflow().getId());
        workflowRepository.save(workflow);

        stateContext.setState(ETrustInvoiceState.IS_REFUNDING);
        stateContext.scheduleExternalEvent(
                workflow.getId(), TRefundCreated.newBuilder().build()
        );
    }
}
