package ru.yandex.travel.orders.workflows.invoice.trust.handlers;

import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Component;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentClear;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentRefund;
import ru.yandex.travel.orders.workflow.invoice.proto.TScheduleClearing;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustInvoiceCallbackReceived;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustPaymentCleared;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@Component
@IgnoreEvents(types = {TPaymentClear.class, TScheduleClearing.class, TTrustInvoiceCallbackReceived.class})
public class ClearingStateHandler extends AnnotatedStatefulWorkflowEventHandler<ETrustInvoiceState, TrustInvoice> {

    @HandleEvent
    public void handleTrustPaymentCleared(TTrustPaymentCleared message,
                                          StateContext<ETrustInvoiceState, TrustInvoice> ctx) {
        TrustInvoice invoice = ctx.getWorkflowEntity();
        invoice.getInvoiceItems().forEach(item -> item.setClearedSum(item.getPrice()));
        ctx.setState(ETrustInvoiceState.IS_CLEARED);
        if( message.hasClearedAt() ) {
            invoice.setTrustClearedAt(ProtoUtils.toInstant(message.getClearedAt()));
        }
        if( message.hasRealClearedAt() ) {
            invoice.setTrustRealClearedAt(ProtoUtils.toInstant(message.getRealClearedAt()));
        }
        ctx.scheduleExternalEvent(invoice.getOrderWorkflowId(), TInvoiceCleared.newBuilder().build());
    }

    @HandleEvent
    public void handleTrustPaymentRefund(TPaymentRefund message,
                                         StateContext<ETrustInvoiceState, TrustInvoice> ctx) {
        TrustInvoice invoice = ctx.getWorkflowEntity();
        // notify order that clearing is in process
        ctx.scheduleExternalEvent(invoice.getOrderWorkflowId(), TClearingInProcess.getDefaultInstance());
    }
}
