package ru.yandex.travel.orders.workflows.invoice.trust.handlers;

import java.math.BigDecimal;
import java.text.MessageFormat;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.InvoiceItem;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.entities.MoneyTransferConfig;
import ru.yandex.travel.orders.entities.ResizeTrustRefund;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.WellKnownAccount;
import ru.yandex.travel.orders.repository.ResizeTrustRefundRepository;
import ru.yandex.travel.orders.services.AccountService;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustHotelsProperties;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.services.payments.model.TrustClearResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCompositeOrderPaymentMarkup;
import ru.yandex.travel.orders.services.payments.model.TrustResizeRequest;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentClear;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentCompleted;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentRefund;
import ru.yandex.travel.orders.workflow.invoice.proto.TRefreshPaymentStatus;
import ru.yandex.travel.orders.workflow.invoice.proto.TScheduleClearing;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustInvoiceCallbackReceived;
import ru.yandex.travel.orders.workflow.order.proto.EInvoiceRefundType;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceRefunded;
import ru.yandex.travel.orders.workflow.trust.refund.proto.ETrustResizeState;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

import static ru.yandex.travel.commons.lang.ComparatorUtils.isEqualByCompareTo;
import static ru.yandex.travel.orders.workflows.orderitem.RefundingUtils.convertTargetFiscalItemsFromProto;
import static ru.yandex.travel.orders.workflows.orderitem.RefundingUtils.convertTargetFiscalItemsMarkupFromProto;

@Service
@RequiredArgsConstructor
@Slf4j
@IgnoreEvents(types = {TRefreshPaymentStatus.class, TTrustInvoiceCallbackReceived.class})
public class HoldStateHandler extends AnnotatedStatefulWorkflowEventHandler<ETrustInvoiceState, TrustInvoice> {
    private final TrustClientProvider trustClientProvider;
    private final AccountService accountService;
    private final TrustHotelsProperties trustHotelProperties;
    private final ResizeTrustRefundRepository resizeTrustRefundRepository;

    @HandleEvent
    public void handlePaymentCompleted(TPaymentCompleted event,
                                       StateContext<ETrustInvoiceState, TrustInvoice> ctx) {
        log.info("Payment completed");
        // TODO брать задержку клиринга для каждой вертикали отдельно.
        //  Сейчас электрички и автобусы обнуляют задержку в ConfirmingStateHandler
        Instant clearTime = Instant.now().plus(trustHotelProperties.getClearingDelay().toMillis(), ChronoUnit.MILLIS);
        ctx.getWorkflowEntity().setClearAt(clearTime);
    }

    @HandleEvent
    public void handlePaymentClear(TPaymentClear event,
                                   StateContext<ETrustInvoiceState, TrustInvoice> ctx) {
        Invoice invoice = ctx.getWorkflowEntity();
        TrustUserInfo userInfo = TrustUserInfoHelper.createUserInfo(invoice);
        // here we ignore the result of operation, as it has only the status fields in it, but trust api mirrors
        // status in http codes, and non-2xx code will lead to exception
        TrustClient trustClient = trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile());
        TrustClearResponse paymentsClearResponse = trustClient.clear(invoice.getPurchaseToken(), userInfo);

        // TODO (mbobrov): think of it, while addressing settlement
//        MoneyTransferConfig transferConfig = MoneyTransferConfig.create();
//        invoice.getOrder().getOrderItems().stream()
//                .flatMap(item -> item.getSettlementItems().stream())
//                .forEach(item -> transferConfig.addTransfer(invoice.getOrder().getAccount().getId(),
//                        item.getAccount().getId(),
//                        item.getMoneyAmount().getNumber().numberValue(BigDecimal.class)));
//        transferConfig.setBaseCurrency(invoice.getOrder().getCurrency());
//        transferConfig.setFxRate(invoice.getOrder().getFxRate());
//        accountService.transferMoney(transferConfig);
        ctx.setState(ETrustInvoiceState.IS_CLEARING);
        ctx.getWorkflowEntity().setNextCheckStatusAt(Instant.now());
        ctx.getWorkflowEntity().setBackgroundJobActive(true);
    }

    @HandleEvent
    public void handleScheduleClear(TScheduleClearing event, StateContext<ETrustInvoiceState, TrustInvoice> ctx) {
        ctx.getWorkflowEntity().setClearAt(ProtoUtils.toInstant(event.getClearAt()));
    }

    @HandleEvent
    public void handlePaymentRefund(TPaymentRefund refund, StateContext<ETrustInvoiceState, TrustInvoice> ctx) {
        TrustInvoice invoice = ctx.getWorkflowEntity();
        TrustUserInfo userInfo = TrustUserInfoHelper.createUserInfo(invoice);
        Map<Long, InvoiceItem> fiscalItemToInvoiceItem = invoice.getInvoiceItems().stream().collect(
                Collectors.toMap(InvoiceItem::getFiscalItemId, Function.identity())
        );
        TrustClient trustClient = trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile());

        Map<Long, Money> targetItems = convertTargetFiscalItemsFromProto(refund.getTargetFiscalItemsMap());
        Map<Long, MoneyMarkup> targetMarkups =
                convertTargetFiscalItemsMarkupFromProto(refund.getTargetFiscalItemsMarkupMap());

        Set<Long> fiscalItemInsideInvoice = fiscalItemToInvoiceItem.keySet();
        Set<Long> fiscalItemsInsideRefundRequest = targetItems.keySet();

        boolean fullRefund = targetItems.values().stream().allMatch(Money::isZero);

        BigDecimal refundSum = BigDecimal.ZERO;

        if (fiscalItemInsideInvoice.equals(fiscalItemsInsideRefundRequest) && fullRefund) {
            trustClient.unhold(invoice.getPurchaseToken(), userInfo);
            ResizeTrustRefund resizeRef = createRefund(refund, invoice);

            for (InvoiceItem invoiceItem : invoice.getInvoiceItems()) {
                refundSum = refundSum.add(invoiceItem.getPrice());
                invoiceItem.setPrice(BigDecimal.ZERO);
                invoiceItem.setClearedSum(BigDecimal.ZERO);
                invoiceItem.setYandexPlusWithdraw(BigDecimal.ZERO);

                Money targetPrice = targetItems.get(invoiceItem.getFiscalItemId());
                MoneyMarkup targetMarkup = targetMarkups.get(invoiceItem.getFiscalItemId());
                addRefundItem(resizeRef, invoiceItem, targetPrice, targetMarkup);
                resizeTrustRefundRepository.save(resizeRef);
            }
        } else {
            for (Map.Entry<Long, Money> entry : targetItems.entrySet()) {
                InvoiceItem invoiceItem = fiscalItemToInvoiceItem.get(entry.getKey());
                if (invoiceItem == null) {
                    throw new RuntimeException(MessageFormat.format("Invoice item with fiscal item id {0} not found " +
                            "for refund", entry.getKey()));
                }
                BigDecimal amount = entry.getValue().getNumberStripped();
                refundSum = refundSum.add(invoiceItem.getPrice().subtract(amount));

                ResizeTrustRefund resizeRef = createRefund(refund, invoice);
                MoneyMarkup newMarkup = targetMarkups.get(invoiceItem.getFiscalItemId());
                addRefundItem(resizeRef, invoiceItem, entry.getValue(), newMarkup);
                resizeTrustRefundRepository.save(resizeRef);

                TrustResizeRequest req = new TrustResizeRequest();
                req.setAmount(amount);
                req.setQty(1);
                addMarkupToResizeRequest(req, invoiceItem.getTrustOrderId(), invoiceItem.getPriceMarkup(), newMarkup);
                trustClient.resize(invoice.getPurchaseToken(), invoiceItem.getTrustOrderId(), req, userInfo);

                updateInvoiceItemResizedValues(invoiceItem, req);
            }
        }

        if (refundSum.compareTo(BigDecimal.ZERO) > 0) {
            invoice.initClearFiscalReceipt(ProtoUtils.fromStringOrNull(refund.getOrderRefundId()));
        }

        TrustBasketStatusResponse trustBasketStatus = trustClient.getBasketStatus(invoice.getPurchaseToken(),
                userInfo);
        InvoiceUtils.checkSums(invoice, trustBasketStatus);

        MoneyTransferConfig moneyTransferConfig = MoneyTransferConfig.create()
                .setBaseCurrency(invoice.getAccount().getCurrency())
                .addTransfer(invoice.getAccount().getId(), WellKnownAccount.TRUST.getUuid(), refundSum);
        accountService.transferMoney(moneyTransferConfig);

        if (invoice.calculateCurrentAmount().compareTo(BigDecimal.ZERO) == 0) {
            ctx.setState(ETrustInvoiceState.IS_CANCELLED);
        }

        ctx.scheduleExternalEvent(
                invoice.getOrderWorkflowId(),
                TInvoiceRefunded.newBuilder()
                        .setOrderRefundId(refund.getOrderRefundId())
                        .setInvoiceId(invoice.getId().toString())
                        .setRefundType(EInvoiceRefundType.EIR_RESIZE)
                        .build()
        );
    }

    private ResizeTrustRefund createRefund(TPaymentRefund refund, Invoice invoice) {
        ResizeTrustRefund resizeRefund = new ResizeTrustRefund();
        resizeRefund.setDescription(refund.getReason());
        resizeRefund.setState(ETrustResizeState.RSS_SUCCESS);
        if (!Strings.isNullOrEmpty(refund.getOrderRefundId())) {
            resizeRefund.setOrderRefundId(UUID.fromString(refund.getOrderRefundId()));
        }
        resizeRefund.setInvoice(invoice);
        return resizeRefund;
    }

    private void addRefundItem(ResizeTrustRefund resizeRefund, InvoiceItem invoiceItem,
                               Money targetAmount, MoneyMarkup targetMarkup) {
        resizeRefund.addRefundItem(
                invoiceItem.getTrustOrderId(),
                invoiceItem.getPrice(),
                targetAmount.getNumberStripped(),
                invoiceItem.getPriceMarkup(),
                targetMarkup
        );
    }

    private void addMarkupToResizeRequest(TrustResizeRequest req, String trustOrderId,
                                          MoneyMarkup currentMarkup, MoneyMarkup resizeMarkup) {
        if (resizeMarkup == null) {
            Preconditions.checkState(currentMarkup.isCardOnly(),
                    "No resize markup for composite item money: %s", currentMarkup);
            return;
        }
        Preconditions.checkArgument(isEqualByCompareTo(req.getAmount(), resizeMarkup.getTotal().getNumberStripped()),
                "Payment markup has to sum up to total amount %s but got %s",
                req.getAmount(), resizeMarkup);

        req.setPaymethodMarkup(Map.of(trustOrderId, InvoiceUtils.toTrustPaymentMarkup(resizeMarkup)));
    }

    private void updateInvoiceItemResizedValues(InvoiceItem item, TrustResizeRequest resize) {
        item.setPrice(resize.getAmount());

        // optional markup
        if (resize.getPaymethodMarkup() == null) {
            return;
        }
        TrustCompositeOrderPaymentMarkup markup = resize.getPaymethodMarkup().values().iterator().next();
        item.setYandexPlusWithdraw(markup.getYandexAccount());
    }
}
