package ru.yandex.travel.orders.workflows.invoice.trust.handlers;

import java.time.Clock;
import java.time.Instant;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.protobuf.Message;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.entities.InvoiceItem;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.services.hotels.HotelOrderDetailsHelpers;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustHotelsProperties;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.TrustXpayProperties;
import ru.yandex.travel.orders.services.payments.model.TrustCompositeOrderPaymentMarkup;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketAfsParams;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketDeveloperPayload;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketOrder;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketPassParams;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketRequest;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketResponse;
import ru.yandex.travel.orders.services.payments.model.TrustCreateBasketTerminalRouteData;
import ru.yandex.travel.orders.services.payments.model.TrustCreateOrderResponse;
import ru.yandex.travel.orders.services.payments.model.TrustStartPaymentResponse;
import ru.yandex.travel.orders.services.plus.YandexPlusPayloadService;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentCreated;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Service("paymentNewStateHandler")
@RequiredArgsConstructor
@Slf4j
public class NewStateHandler extends AnnotatedStatefulWorkflowEventHandler<ETrustInvoiceState, TrustInvoice> {

    private final TrustHotelsProperties trustConfig;
    private final TrustXpayProperties xpayConfig;
    // should remove it and make a serialization annotation for dev payload instead
    private final ObjectMapper objectMapper = new ObjectMapper();
    private final Clock clock;
    private final TrustClientProvider trustClientProvider;
    private final YandexPlusPayloadService plusPayloadService;


    @HandleEvent
    public void handlePaymentCreated(TPaymentCreated message,
                                     StateContext<ETrustInvoiceState, TrustInvoice> stateContext) {
        TrustInvoice invoice = stateContext.getWorkflowEntity();

        TrustUserInfo userInfo = new TrustUserInfo(
                invoice.getPassportId(),
                invoice.getUserIp()
        );

        TrustClient trustClient = trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile());

        invoice.getInvoiceItems().forEach(ii -> {
            TrustCreateOrderResponse rsp = trustClient.createOrder(ii.getFiscalItemType().getTrustId(), userInfo);
            ii.setTrustOrderId(rsp.getOrderId());
        });

        Message paymentTestContext = invoice.getEffectivePaymentTestContext();

        TrustCreateBasketResponse basketResponse = trustClient.createBasket(buildCreateRequest(invoice), userInfo,
                paymentTestContext);
        invoice.setPurchaseToken(basketResponse.getPurchaseToken());

        TrustStartPaymentResponse startResponse = trustClient.startPayment(invoice.getPurchaseToken(), userInfo);

        invoice.setPaymentUrl(startResponse.getPaymentUrl());
        stateContext.setState(ETrustInvoiceState.IS_WAIT_FOR_PAYMENT);
        stateContext.scheduleEvent(TPaymentStarted.newBuilder().build());
        stateContext.scheduleExternalEvent(invoice.getOrderWorkflowId(), TInvoicePaymentStarted.newBuilder().build());
    }

    private TrustCreateBasketRequest buildCreateRequest(TrustInvoice invoice) {
        TrustCreateBasketRequest request = new TrustCreateBasketRequest();
        request.setPaymethodId(invoice.getRequestedPaymentMethodId());

        request.setTemplateTag(invoice.getSource());
        if (StringUtils.isNotEmpty(invoice.getClientToken())) {
            request.setToken(invoice.getClientToken());
        } else {
            request.setUid(invoice.getPassportId());
        }

        request.setCurrency("RUB");
        request.setBackUrl(trustConfig.getCallbackUrl());
        request.setReturnPath(invoice.getReturnPath());

        long timeout;
        if (invoice.getExpirationDate() != null && Instant.now(clock).isBefore(invoice.getExpirationDate())) {
            timeout = Instant.now(clock).until(invoice.getExpirationDate(), ChronoUnit.SECONDS);
        } else {
            timeout = trustConfig.getDefaultPaymentTimeout().toSeconds();
        }
        request.setPaymentTimeout((double) timeout);
        request.setWaitForCvn("0"); // is set to 1 only in cases when cvn is provided via backend

        request.setFiscalTaxationType(trustConfig.getTaxationType());

        request.setFiscalNds(invoice.getFiscalNds());
        request.setFiscalTitle(invoice.getFiscalTitle());

        request.setKeepToken("0");

        request.setUserEmail(invoice.getClientEmail());

        String cleanedPhone = invoice.getClientPhone().replaceAll("[^\\d]", "");
        request.setUserPhone(cleanedPhone.substring(0, Math.min(15, cleanedPhone.length())));


        // REQUIRE RUB SETTLEMENT
        if (invoice.getRequireRubSettlement() != null && invoice.getRequireRubSettlement()) {
            getOrCreatePassParams(request).setRubSettlement(1);
        }

        // FORCE 3DS PROPERTY
        if (invoice.getForceThreeDs() == null || invoice.getForceThreeDs()) {
            getOrCreateTerminalRouteData(request).setServiceForce3ds(1);
        }

        // Set terminal for partner
        // https://st.yandex-team.ru/RASPFRONT-9821
        if (invoice.getTerminalForPartner() != null) {
            getOrCreateTerminalRouteData(request).setTerminalForPartner(invoice.getTerminalForPartner());
        }

        // MIR routing
        if (invoice.getUseMirPromo() != null && invoice.getUseMirPromo()) {
            getOrCreatePassParams(request).setMirPromo(1);
        }

        // Plus payload
        if (invoice.calculateCurrentAmountMarkup().getYandexAccount().isPositive()) {
            getOrCreatePassParams(request).setPayload(
                    plusPayloadService.createWithdrawPayloadForService(invoice.getOrder()));
        }

        setDeveloperPayloadIfNeeded(invoice, request);

        request.setOrders(new ArrayList<>());
        Map<String, MoneyMarkup> moneyMarkups = new HashMap<>();
        boolean cardOnly = invoice.getInvoiceItems().stream()
                .map(InvoiceItem::getPriceMarkup)
                .allMatch(MoneyMarkup::isCardOnly);
        for (InvoiceItem item : invoice.getInvoiceItems()) {
            Preconditions.checkState(
                    !Strings.isNullOrEmpty(item.getTrustOrderId()),
                    "Trust order not found for invoice item with id: %s", item.getId()
            );
            TrustCreateBasketOrder order = TrustCreateBasketOrder.builder()
                    .orderId(item.getTrustOrderId())
                    .price(item.getPrice())
                    .qty(1)
                    .fiscalNds(item.getFiscalNds().getTrustValue())
                    .fiscalTitle(item.getFiscalTitle())
                    .fiscalInn(item.getFiscalInn())
                    .build();
            MoneyMarkup markup = item.getPriceMarkup();
            if (!cardOnly) {
                if (moneyMarkups.containsKey(order.getOrderId())) {
                    moneyMarkups.put(order.getOrderId(), moneyMarkups.get(order.getOrderId()).add(markup));
                } else {
                    moneyMarkups.put(order.getOrderId(), markup);
                }
            }
            request.getOrders().add(order);
        }

        if (!moneyMarkups.isEmpty()) {
            Map<String, TrustCompositeOrderPaymentMarkup> paymentMarkups = new HashMap<>();
            moneyMarkups.forEach((id, markup) -> paymentMarkups.put(id, InvoiceUtils.toTrustPaymentMarkup(markup).withoutZeros()));
            request.setPaymethodMarkup(paymentMarkups);
        }

        setAfsParamsIfNecessary(request, invoice);

        return request;
    }

    private TrustCreateBasketPassParams getOrCreatePassParams(TrustCreateBasketRequest request) {
        if (request.getPassParams() == null) {
            request.setPassParams(new TrustCreateBasketPassParams());
        }
        return request.getPassParams();
    }

    private TrustCreateBasketTerminalRouteData getOrCreateTerminalRouteData(TrustCreateBasketRequest request) {
        TrustCreateBasketPassParams passParams = getOrCreatePassParams(request);
        if (passParams.getTerminalRouteData() == null) {
            passParams.setTerminalRouteData(new TrustCreateBasketTerminalRouteData());
        }
        return passParams.getTerminalRouteData();
    }

    private void setAfsParamsIfNecessary(TrustCreateBasketRequest request, TrustInvoice invoice) {
        Optional.ofNullable(invoice.getOrder())
                .filter(order ->
                        OrderCompatibilityUtils.isTrainOrder(order) ||
                                OrderCompatibilityUtils.isHotelOrder(order))
                .map(order -> {
                    var afsParams =
                            TrustCreateBasketAfsParams.builder()
                                    .email(invoice.getClientEmail())
                                    .ip(invoice.getUserIp())
                                    .orderId(order.getPrettyId());

                    if (OrderCompatibilityUtils.isHotelOrder(order)) {
                        HotelItinerary itinerary = OrderCompatibilityUtils.getOnlyHotelOrderItem(order)
                                .getHotelItinerary();

                        Guest firstGuest = itinerary.getGuests().get(0);
                        afsParams.guestName(firstGuest.getFullName());

                        afsParams.hotelName(itinerary.getOrderDetails().getHotelName());
                        Optional.ofNullable(HotelOrderDetailsHelpers.getHotelCountry(itinerary.getOrderDetails()))
                                .ifPresent(geo -> afsParams.hotelCountry(geo.getName()));
                        Optional.ofNullable(HotelOrderDetailsHelpers.getHotelCity(itinerary.getOrderDetails()))
                                .ifPresent(geo -> afsParams.hotelCity(geo.getName()));
                    }

                    return afsParams.build();
                })
                .ifPresent(request::setAfsParams);
    }

    private void setDeveloperPayloadIfNeeded(TrustInvoice invoice, TrustCreateBasketRequest request) {
        TrustCreateBasketDeveloperPayload developerPayload = null;

        // ProcessThroughYt flag for new trains billing scheme
        if (trustConfig.isTrainsNewProcessingEnabled() && invoice.getProcessThroughYt() != null) {
            developerPayload = new TrustCreateBasketDeveloperPayload();
            developerPayload.setProcessThroughYt(invoice.getProcessThroughYt() ? 1 : 0);
        }

        if (developerPayload == null) {
            developerPayload = new TrustCreateBasketDeveloperPayload();
        }
        // todo(alexcrush,tlg-13): a temporary workaround for the broken suburban booking flow,
        // we need to refactor this form choice logic
        Order order = invoice.getOrder();
        EDisplayOrderType orderType = order != null ? order.getDisplayType() : null;
        String paymentFormTemplateName = trustConfig.getSettings(orderType).getPaymentWebFormTemplateName();
        if (invoice.useNewCommonPaymentWebForm()) {
            paymentFormTemplateName = trustConfig.getSettings(orderType).getCommonPaymentWebForm();
            developerPayload.setPaymentCompletionAction("redirect");
            developerPayload.setEnablePaymentCancellation(true);
        }
        developerPayload.setTemplate(Strings.emptyToNull(paymentFormTemplateName));

        String purchaseLabel = xpayConfig.getPurchaseLabel();
        if (!Strings.isNullOrEmpty(purchaseLabel)) {
            developerPayload.setPurchaseLabel(purchaseLabel);
        }

        try {
            String developerPayloadStr = objectMapper.writeValueAsString(developerPayload);
            request.setDeveloperPayload(developerPayloadStr);
        } catch (JsonProcessingException e) {
            log.error("Couldn't serialize trust developer payload: " + developerPayload, e);
        }
    }
}
