package ru.yandex.travel.orders.workflows.invoice.trust.handlers;

import java.math.BigDecimal;
import java.util.Map;
import java.util.Objects;
import java.util.function.Function;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.InvoiceItem;
import ru.yandex.travel.orders.entities.MoneyMarkup;
import ru.yandex.travel.orders.entities.MoneyTransferConfig;
import ru.yandex.travel.orders.entities.SimpleTrustRefund;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.TrustRefundItem;
import ru.yandex.travel.orders.entities.WellKnownAccount;
import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.services.AccountService;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustUserInfo;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TRefreshPaymentStatus;
import ru.yandex.travel.orders.workflow.invoice.proto.TRefundError;
import ru.yandex.travel.orders.workflow.invoice.proto.TRefundFailed;
import ru.yandex.travel.orders.workflow.invoice.proto.TRefundSuccess;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustInvoiceCallbackReceived;
import ru.yandex.travel.orders.workflow.order.proto.EInvoiceRefundType;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceNotRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceRefunded;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.TWorkflowCrashed;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Service
@Slf4j
@IgnoreEvents(types = {TRefreshPaymentStatus.class, TTrustInvoiceCallbackReceived.class})
@RequiredArgsConstructor
public class RefundingStateHandler extends AnnotatedStatefulWorkflowEventHandler<ETrustInvoiceState, TrustInvoice> {
    private final AccountService accountService;

    private final TrustClientProvider trustClientProvider;

    private final StarTrekService starTrekService;

    @HandleEvent
    public void handleRefundSuccess(TRefundSuccess message,
                                    StateContext<ETrustInvoiceState, TrustInvoice> stateContext) {
        // TODO (mbobrov) - change account info
        // think of going to refund state in case of all possible money refunded
        TrustInvoice invoice = stateContext.getWorkflowEntity();
        TrustUserInfo userInfo = TrustUserInfoHelper.createUserInfo(invoice);
        TrustClient trustClient = trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile());

        TrustBasketStatusResponse paymentStatus = trustClient.getBasketStatus(invoice.getPurchaseToken(), userInfo);

        SimpleTrustRefund activeRefund = stateContext.getWorkflowEntity().getActiveTrustRefund();
        MoneyTransferConfig moneyTransferConfig = MoneyTransferConfig.create()
            .setBaseCurrency(stateContext.getWorkflowEntity().getAccount().getCurrency())
            .addTransfer(
                stateContext.getWorkflowEntity().getAccount().getId(), WellKnownAccount.TRUST.getUuid(),
                activeRefund.getTotalRefundSum()
            );
        accountService.transferMoney(moneyTransferConfig);

        invoice.addRefundFiscalReceipt(activeRefund.getTrustRefundId(), activeRefund.getOrderRefundId());

        Map<String, InvoiceItem> trustOrderIdToInvoiceItem = invoice.getInvoiceItems().stream().collect(
            Collectors.toMap(InvoiceItem::getTrustOrderId, Function.identity())
        );

        for (TrustRefundItem ri : activeRefund.getRefundItems()) {
            InvoiceItem invoiceItem = trustOrderIdToInvoiceItem.get(ri.getTrustOrderId());
            MoneyMarkup currentMarkup = invoiceItem.getPriceMarkup();
            invoiceItem.setPrice(invoiceItem.getPrice().subtract(ri.getRefundDelta()));

            BigDecimal alreadyRefunded = Objects.requireNonNullElse(invoiceItem.getRefundedSum(), BigDecimal.ZERO);
            invoiceItem.setRefundedSum(alreadyRefunded.add(ri.getRefundDelta()));

            MoneyMarkup refundMarkup = ri.getRefundDeltaMarkup();
            MoneyMarkup newMarkup = currentMarkup.subtract(refundMarkup);
            Preconditions.checkArgument(!newMarkup.hasNegativeValues(),
                    "Applying refund causes negative values; current %s, refunded %s", currentMarkup, refundMarkup);
            if (!refundMarkup.getYandexAccount().isZero()) {
                invoiceItem.setYandexPlusWithdrawMoney(newMarkup.getYandexAccount());
            }
        }

        InvoiceUtils.checkSums(invoice, paymentStatus); // check that the sums are correct

        if (invoice.calculateCurrentAmount().compareTo(BigDecimal.ZERO) == 0) {
            stateContext.setState(ETrustInvoiceState.IS_REFUNDED);
        } else {
            stateContext.setState(ETrustInvoiceState.IS_CLEARED);
        }

        stateContext.scheduleExternalEvent(invoice.getOrderWorkflowId(), TInvoiceRefunded.newBuilder()
                .setOrderRefundId(ProtoUtils.toStringOrEmpty(activeRefund.getOrderRefundId()))
                .setInvoiceId(invoice.getId().toString())
                .setRefundType(EInvoiceRefundType.EIR_REFUND)
                .build());
    }

    @HandleEvent
    public void handleRefundFailed(TRefundFailed message,
                                   StateContext<ETrustInvoiceState, TrustInvoice> stateContext) {
        // refund failed, but can be retried, going back to clear state
        SimpleTrustRefund activeRefund = stateContext.getWorkflowEntity().getActiveTrustRefund();
        TrustInvoice invoice = stateContext.getWorkflowEntity();
        stateContext.setState(ETrustInvoiceState.IS_CLEARED);
        stateContext.scheduleExternalEvent(invoice.getOrderWorkflowId(), TInvoiceNotRefunded.newBuilder()
            .setOrderRefundId(ProtoUtils.toStringOrEmpty(activeRefund.getOrderRefundId()))
            .setInvoiceId(invoice.getId().toString()).build());
    }

    @HandleEvent
    public void handleRefundError(TRefundError message,
                                  StateContext<ETrustInvoiceState, TrustInvoice> stateContext) {
        SimpleTrustRefund activeRefund = stateContext.getWorkflowEntity().getActiveTrustRefund();
        starTrekService.createIssueForTrustRefundError(stateContext.getWorkflowEntity(), activeRefund.getId(),
            stateContext, "refund error");
    }

    @HandleEvent
    public void handleRefundError(TWorkflowCrashed message,
                                  StateContext<ETrustInvoiceState, TrustInvoice> stateContext) {
        if (WellKnownWorkflowEntityType.TRUST_REFUND.equalsValue(message.getEntityType())) {
            starTrekService.createIssueForTrustRefundError(stateContext.getWorkflowEntity(),
                ProtoUtils.fromStringOrNull(message.getEntityId()), stateContext, "refund crashed");
        } else {
            throw new RuntimeException(String.format("TWorkflowCrashed received with entityType: %s. Invoice id: %s",
                message.getEntityType(), stateContext.getWorkflowEntity().getId()));
        }
    }
}
