package ru.yandex.travel.orders.workflows.invoice.trust.handlers;

import java.math.BigDecimal;
import java.time.Duration;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.InvoiceItem;
import ru.yandex.travel.orders.entities.MoneyTransferConfig;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.WellKnownAccount;
import ru.yandex.travel.orders.services.AccountService;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.model.PaymentStatusEnum;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentCompleted;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentCompletedWithError;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentStarted;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustInvoiceCallbackReceived;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustPaymentAuthorized;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustPaymentNotAuthorized;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustPaymentStatusChanged;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquireErrorOccurred;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Service
@RequiredArgsConstructor
@Slf4j
public class WaitForPaymentStateHandler extends AnnotatedStatefulWorkflowEventHandler<ETrustInvoiceState,
        TrustInvoice> {

    private final AccountService accountService;

    private final TrustClientProvider trustClientProvider;

    @HandleEvent
    public void handlePaymentStarted(TPaymentStarted message,
                                     StateContext<ETrustInvoiceState, TrustInvoice> ctx) {
        log.info("Payment started for invoice " + ctx.getWorkflowEntity().getId().toString());
        scheduleInvoiceRefresh(ctx);
    }

    private void scheduleInvoiceRefresh(StateContext<ETrustInvoiceState, TrustInvoice> ctx) {
        ctx.getWorkflowEntity().setBackgroundJobActive(true);
        // todo(tlg-13): inject properties and use trust-invoice-refresh-timeout here / see TrustInvoiceRefreshService
        ctx.getWorkflowEntity().rescheduleNextCheckStatusAt(Duration.ofSeconds(3));
    }

    @HandleEvent
    public void handleTrustPaymentAuthorized(TTrustPaymentAuthorized event,
                                             StateContext<ETrustInvoiceState, TrustInvoice> ctx) {

        var invoice = ctx.getWorkflowEntity();
        var basketStatus = ProtoUtils.fromTJson(event.getBasketStatus(), TrustBasketStatusResponse.class);

        handleBasketStatusForInvoice(invoice, basketStatus, ctx);
    }

    @HandleEvent
    public void handleTrustPaymentNotAuthorized(TTrustPaymentNotAuthorized event,
                                                StateContext<ETrustInvoiceState, TrustInvoice> ctx) {
        var invoice = ctx.getWorkflowEntity();
        var basketStatus = ProtoUtils.fromTJson(event.getBasketStatus(), TrustBasketStatusResponse.class);

        handleBasketStatusForInvoice(invoice, basketStatus, ctx);
    }

    @HandleEvent
    public void handleTrustPaymentStatusChanged(TTrustPaymentStatusChanged event,
                                                StateContext<ETrustInvoiceState, TrustInvoice> ctx) {
        var invoice = ctx.getWorkflowEntity();
        var basketStatus = ProtoUtils.fromTJson(event.getBasketStatus(), TrustBasketStatusResponse.class);

        handleBasketStatusForInvoice(invoice, basketStatus, ctx);
    }

    @HandleEvent
    public void handleTrustCallbackReceived(TTrustInvoiceCallbackReceived event,
                                            StateContext<ETrustInvoiceState, TrustInvoice> ctx) {
        log.info("Check basket status because of trust callback for invoice: {}", ctx.getWorkflowEntity().getId());
        var invoice = ctx.getWorkflowEntity();
        // todo(tlg-13): not ready for this change yet but should move there eventually
//        Preconditions.checkArgument(!invoice.isBackgroundJobActive(),
//                "background jobs should have been stopped in order to prevent data races");
//        Preconditions.checkArgument(invoice.getNextCheckStatusAt() == null,
//                "background checks should have been stopped in order to prevent data races");
        TrustBasketStatusResponse basketStatus = trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile())
                .getBasketStatus(invoice.getPurchaseToken(), TrustUserInfoHelper.createUserInfo(invoice));

        if (checkCallbackStatusEqualsPaymentStatus(event.getStatus(), basketStatus.getPaymentStatus())) {
            invoice.setBackgroundJobActive(false);
            invoice.setNextCheckStatusAt(null);

            handleBasketStatusForInvoice(invoice, basketStatus, ctx);
        } else {
            log.warn("Callback status doesn't match actual payment status: " +
                    "callback status is {} but the api one is {}; keeping polling",
                    event.getStatus(), basketStatus.getPaymentStatus());
            scheduleInvoiceRefresh(ctx);
        }
    }

    private boolean checkCallbackStatusEqualsPaymentStatus(String callbackStatus, PaymentStatusEnum paymentStatus) {
        switch (paymentStatus) {
            case AUTHORIZED:
                return "success".equals(callbackStatus);
            case NOT_AUTHORIZED:
                return "cancelled".equals(callbackStatus);
            default:
                return false;
        }
    }

    private void handleBasketStatusForInvoice(TrustInvoice invoice, TrustBasketStatusResponse basketStatus,
                                              StateContext<ETrustInvoiceState, TrustInvoice> ctx) {
        invoice.setTrustPaymentId(basketStatus.getTrustPaymentId());
        invoice.setRrn(basketStatus.getRrn());
        invoice.setUserAccount(basketStatus.getUserAccount());
        switch (basketStatus.getPaymentStatus()) {
            case AUTHORIZED:
                invoice.initAcquireFiscalReceipt();
                ctx.setState(ETrustInvoiceState.IS_HOLD);
                ctx.scheduleEvent(TPaymentCompleted.newBuilder().build());
                if (basketStatus.getStartTs() != null) {
                    invoice.setPaymentStartTs(basketStatus.getStartTs());
                }

                BigDecimal invoiceSum = invoice.getInvoiceItems().stream()
                        .map(InvoiceItem::getPrice)
                        .reduce(BigDecimal::add)
                        .orElseThrow(() -> new IllegalStateException("Cannot compute summary price of invoices: no price present"));
                accountService.transferMoney(MoneyTransferConfig.create()
                        .setBaseCurrency(invoice.getAccount().getCurrency())
                        .addTransfer(WellKnownAccount.TRUST.getUuid(), invoice.getAccount().getId(), invoiceSum));
                ctx.scheduleExternalEvent(invoice.getOrderWorkflowId(), TMoneyAcquired.newBuilder().build());

                invoice.setApprovalCode(basketStatus.getApprovalCode());
                if (basketStatus.getPaymentTs() != null) {
                    invoice.setTrustPaymentTs(basketStatus.getPaymentTs());
                }
                invoice.setCardType(basketStatus.getCardType());
                invoice.setActualPaymentMethodId(basketStatus.getPaymethodId());
                break;
            case NOT_AUTHORIZED:
                log.error("Payment for invoice {} not authorized. Failure reason: {}", invoice.getId(),
                        invoice.getAuthorizationErrorCode());
                ctx.setState(ETrustInvoiceState.IS_PAYMENT_NOT_AUTHORIZED);
                ctx.scheduleEvent(TPaymentCompletedWithError.newBuilder().build());
                ctx.scheduleExternalEvent(ctx.getWorkflowEntity().getOrderWorkflowId(),
                        TMoneyAcquireErrorOccurred.newBuilder().build());
                if (!Strings.isNullOrEmpty(basketStatus.getPaymentRespCode())) {
                    invoice.setAuthorizationErrorCode(basketStatus.getPaymentRespCode());
                    invoice.setAuthorizationErrorDesc(basketStatus.getPaymentRespDesc());
                }
                if (basketStatus.getStartTs() != null) {
                    invoice.setPaymentStartTs(basketStatus.getStartTs());
                }
                if (basketStatus.getCancelTs() != null) {
                    invoice.setPaymentCancelTs(basketStatus.getCancelTs());
                }
                break;
            default:
                break;
        }
    }
}
