package ru.yandex.travel.orders.workflows.invoice.trust.jobs;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.repository.TrustInvoiceRepository;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TPaymentClear;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Service
@Slf4j
public class ClearingService {
    private final TrustInvoiceRepository trustInvoiceRepository;
    private final WorkflowMessageSender workflowMessageSender;

    @Autowired
    public ClearingService(TrustInvoiceRepository trustInvoiceRepository,
                           WorkflowMessageSender workflowMessageSender) {
        this.trustInvoiceRepository = trustInvoiceRepository;
        this.workflowMessageSender = workflowMessageSender;
    }

    @Transactional
    public List<UUID> fetchInvoicesToClear(Set<UUID> activeInvoices, int maxResultSize) {
        return trustInvoiceRepository.getInvoiceIdsWaitingClearingWithExclusions(Instant.now(),
                ETrustInvoiceState.IS_HOLD, EWorkflowState.WS_RUNNING, activeInvoices,
                PageRequest.of(0, maxResultSize));
    }

    @Transactional
    public long getInvoicesToScheduleClearingCount(Set<UUID> activeInvoices) {
        return trustInvoiceRepository.countInvoicesWaitingForClearanceWithExclusions(
                Instant.now(), ETrustInvoiceState.IS_HOLD, EWorkflowState.WS_RUNNING, activeInvoices);
    }

    @TransactionMandatory
    public void clearInvoice(UUID invoiceId) {
        TrustInvoice invoice = trustInvoiceRepository.getOne(invoiceId);
        clearInvoice(invoice);
    }

    @TransactionMandatory
    public void clearInvoice(TrustInvoice invoice) {
        // we set the field only to cause OptimisticLockException in case of concurrent invoice updates
        invoice.setClearingScheduledAt(Instant.now());
        workflowMessageSender.scheduleEvent(invoice.getWorkflow().getId(), TPaymentClear.newBuilder().build());
    }
}
