package ru.yandex.travel.orders.workflows.invoice.trust.jobs;

import java.time.Instant;
import java.util.List;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.entities.FiscalReceipt;
import ru.yandex.travel.orders.entities.FiscalReceiptState;
import ru.yandex.travel.orders.repository.FiscalReceiptRepository;
import ru.yandex.travel.orders.services.payments.TrustClient;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.workflow.exceptions.RetryableException;

@Service
@RequiredArgsConstructor
@Slf4j
@EnableConfigurationProperties(FiscalReceiptProperties.class)
public class FiscalReceiptService {

    private final TrustClientProvider trustClientProvider;

    private final FiscalReceiptRepository fiscalReceiptRepository;

    private final FiscalReceiptRefreshDelayService refreshDelayService;

    private final FiscalReceiptProperties fiscalReceiptProperties;

    public List<Long> fetchPendingFiscalReceiptIds(int maxResultSize) {
        return fiscalReceiptRepository.getIdsOfFiscalReceiptsWithNextRefreshBeforeAndInState(Instant.now(),
                FiscalReceiptState.CREATED,
                PageRequest.of(0, maxResultSize));
    }

    public long getPendingFiscalReceiptCount() {
        return fiscalReceiptRepository.countFiscalReceiptsWithNextRefreshBeforeAndInState(Instant.now(),
                FiscalReceiptState.CREATED);
    }

    public void fetchFiscalReceipt(Long fiscalReceiptId) {
        FiscalReceipt fiscalReceipt = fiscalReceiptRepository.getOne(fiscalReceiptId);
        try (var ignored = NestedMdc.forEntity(fiscalReceipt.getInvoice())) {
            log.info("Fetching fiscal receipt (id: {}, type: {}, parent id: {})",
                    fiscalReceipt.getId(), fiscalReceipt.getReceiptType(), fiscalReceipt.getParentId());
            TrustClient trustClient =
                    trustClientProvider.getTrustClientForPaymentProfile(fiscalReceipt.getPaymentProfile());
            switch (fiscalReceipt.getReceiptType()) {
                case ACQUIRE:
                    refreshFiscalReceipt(fiscalReceipt,
                            trustClient.getBasketStatus(fiscalReceipt.getParentId(), null).getFiscalReceiptUrl());
                    break;
                case CLEAR:
                    refreshFiscalReceipt(fiscalReceipt,
                            trustClient.getBasketStatus(fiscalReceipt.getParentId(), null).getFiscalReceiptClearingUrl());
                    break;
                case REFUND:
                    refreshFiscalReceipt(fiscalReceipt,
                            trustClient.getRefundStatus(fiscalReceipt.getParentId(), null).getFiscalReceiptUrl());
                    break;
            }
        } catch (RetryableException e) {
            // do nothing
        } catch (Exception e) {
            refreshFiscalReceipt(fiscalReceipt, null);
        }
    }

    private void refreshFiscalReceipt(FiscalReceipt fiscalReceipt, String receiptUrl) {
        if (Strings.isNullOrEmpty(receiptUrl)) {
            int attempt = fiscalReceipt.getNextRefreshAttempt();
            if (attempt < fiscalReceiptProperties.getMaxRefreshAttempts()) {
                fiscalReceipt.rescheduleNextRefresh(refreshDelayService.calculateNextDelay(attempt));
            } else {
                fiscalReceipt.resetNextRefresh();
                fiscalReceipt.setState(FiscalReceiptState.ERROR);
            }
        } else {
            log.info("Got fiscal receipt for (id: {}, type: {}, url: {})", fiscalReceipt.getId(),
                    fiscalReceipt.getReceiptType(), receiptUrl);
            fiscalReceipt.setReceiptUrl(receiptUrl);
            fiscalReceipt.resetNextRefresh();
            fiscalReceipt.setState(FiscalReceiptState.FETCHED);
        }
    }
}
