package ru.yandex.travel.orders.workflows.invoice.trust.jobs;

import java.time.Duration;
import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.orders.services.payments.TrustHotelsProperties;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorProperties;

@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties({TrustHotelsProperties.class, FiscalReceiptProperties.class})
public class TaskProcessorsConfiguration {

    private final TrustInvoiceRefreshService trustInvoiceRefreshService;

    private final FiscalReceiptService fiscalReceiptService;

    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;

    private final TrustHotelsProperties trustHotelsProperties;

    private final FiscalReceiptProperties fiscalReceiptProperties;

    private final ClearingService clearingService;

    @Bean
    @ConfigurationProperties(value = "trust-await-payment-processor", ignoreUnknownFields = false)
    public TaskProcessorProperties trustInvoiceAwaitingPaymentProcessorProperties() {
        return TaskProcessorProperties.builder()
                .name("TrustAwaitPayment")
                .build();
    }

    @Bean
    public TaskProcessor<UUID> trustInvoiceAwaitingPaymentProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return trustInvoiceRefreshService.fetchInvoicesWaitingAuthorization(getLockedTaskKeys(), maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return trustInvoiceRefreshService.getInvoicesWaitingAuthorizationCount(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("TrustAwaitingPayment");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, trustInvoiceRefreshService::refreshInvoiceWaitingPayment,
                forcedRollbackTxManagerWrapper, txDefinition,
                trustInvoiceAwaitingPaymentProcessorProperties());
    }

    @Bean
    public TaskProcessor<UUID> trustInvoiceAwaitingClearingProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return trustInvoiceRefreshService.fetchInvoicesWaitingClearing(getLockedTaskKeys(), maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return trustInvoiceRefreshService.getInvoicesWaitingClearingCount(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("TrustAwaitingClearing");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, trustInvoiceRefreshService::refreshInvoiceWaitingClearing,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("TrustAwaitClearing").daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(trustHotelsProperties.getNumberOfInvoicesToClear())
                        .initialStartDelay(Duration.ofMillis(50)).scheduleRate(Duration.ofSeconds(1)).build());
    }

    @Bean
    public TaskProcessor<Long> trustFetchFiscalReceiptsProcessor() {
        TaskKeyProvider<Long> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<Long> getPendingTaskKeys(int maxResultSize) {
                return fiscalReceiptService.fetchPendingFiscalReceiptIds(maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return fiscalReceiptService.getPendingFiscalReceiptCount();
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("TrustFetchFiscalReceipt");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, fiscalReceiptService::fetchFiscalReceipt,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("TrustFetchFiscalReceipt").daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(fiscalReceiptProperties.getNumberOfReceiptsToRefresh())
                        .initialStartDelay(Duration.ofMillis(50)).scheduleRate(Duration.ofSeconds(1)).build());
    }

    @Bean
    public TaskProcessor<UUID> trustScheduleClearingTaskProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return clearingService.fetchInvoicesToClear(getLockedTaskKeys(), maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return clearingService.getInvoicesToScheduleClearingCount(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("TrustScheduleClearing");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, clearingService::clearInvoice,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("TrustScheduleClearing").daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(trustHotelsProperties.getNumberOfInvoicesToClear())
                        .initialStartDelay(Duration.ofMillis(100))
                        .scheduleRate(trustHotelsProperties.getScheduleClearingRate()).build());
    }

}
