package ru.yandex.travel.orders.workflows.invoice.trust.jobs;

import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.repository.TrustInvoiceRepository;
import ru.yandex.travel.orders.services.payments.TrustClientProvider;
import ru.yandex.travel.orders.services.payments.TrustHotelsProperties;
import ru.yandex.travel.orders.services.payments.model.TrustBasketStatusResponse;
import ru.yandex.travel.orders.workflow.invoice.proto.ETrustInvoiceState;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustPaymentCleared;
import ru.yandex.travel.orders.workflow.invoice.proto.TTrustPaymentStatusChanged;
import ru.yandex.travel.orders.workflows.invoice.trust.handlers.TrustUserInfoHelper;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowMessageSender;

@Service
@Slf4j
@RequiredArgsConstructor
@EnableConfigurationProperties(TrustHotelsProperties.class)
public class TrustInvoiceRefreshService {

    private final TrustInvoiceRepository trustInvoiceRepository;

    private final WorkflowMessageSender workflowMessageSender;

    private final TrustClientProvider trustClientProvider;

    private final TrustHotelsProperties trustHotelsProperties;

    @TransactionMandatory
    public List<UUID> fetchInvoicesWaitingAuthorization(Set<UUID> activeInvoices, int maxResultSize) {
        return trustInvoiceRepository.getInvoiceIdsAwaitingRefreshInStateWithExclusions(Instant.now(),
                ETrustInvoiceState.IS_WAIT_FOR_PAYMENT, EWorkflowState.WS_RUNNING,
                activeInvoices, PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    public long getInvoicesWaitingAuthorizationCount(Set<UUID> activeInvoices) {
        return trustInvoiceRepository.countInvoicesAwaitingRefreshInStateWithExclusions(Instant.now(),
                ETrustInvoiceState.IS_WAIT_FOR_PAYMENT, EWorkflowState.WS_RUNNING, activeInvoices);
    }

    @TransactionMandatory
    public List<UUID> fetchInvoicesWaitingClearing(Set<UUID> activeInvoices, int maxResultSize) {
        return trustInvoiceRepository.getInvoiceIdsAwaitingRefreshInStateWithExclusions(Instant.now(),
                ETrustInvoiceState.IS_CLEARING, EWorkflowState.WS_RUNNING, activeInvoices,
                PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    public long getInvoicesWaitingClearingCount(Set<UUID> activeInvoices) {
        return trustInvoiceRepository.countInvoicesAwaitingRefreshInStateWithExclusions(Instant.now(),
                ETrustInvoiceState.IS_CLEARING, EWorkflowState.WS_RUNNING, activeInvoices);
    }

    @TransactionMandatory
    public void refreshInvoiceWaitingPayment(UUID invoiceId) {
        TrustInvoice invoice = trustInvoiceRepository.getOne(invoiceId);
        try (var ignored = NestedMdc.forEntity(invoice.getId(), invoice.getEntityType())) {
            if (!invoice.isBackgroundJobActive()) {
                log.info("Background job not active, returning");
                return;
            }

            TrustBasketStatusResponse trustBasketStatusResponse =
                    trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile()).getBasketStatus(
                            invoice.getPurchaseToken(), TrustUserInfoHelper.createUserInfo(invoice)
                    );

            switch (trustBasketStatusResponse.getPaymentStatus()) {
                case AUTHORIZED:
                case NOT_AUTHORIZED:
                    invoice.setBackgroundJobActive(false);
                    invoice.setNextCheckStatusAt(null);
                    workflowMessageSender.scheduleEvent(invoice.getWorkflow().getId(),
                            TTrustPaymentStatusChanged.newBuilder()
                                    .setBasketStatus(ProtoUtils.toTJson(trustBasketStatusResponse))
                                    .build());
                    break;
                default:
                    invoice.rescheduleNextCheckStatusAt(trustHotelsProperties.getPaymentRefreshTimeout());
                    break;
            }
        } catch (Exception e) {
            log.error("Exception getting trust status while waiting payment", e);
            invoice.rescheduleNextCheckStatusAt(trustHotelsProperties.getPaymentRefreshTimeout());
        }
    }

    @TransactionMandatory
    public void refreshInvoiceWaitingClearing(UUID invoiceId) {
        TrustInvoice invoice = trustInvoiceRepository.getOne(invoiceId);
        try (var ignored = NestedMdc.forEntity(invoice.getId(), invoice.getEntityType())) {
            if (!invoice.isBackgroundJobActive()) {
                log.info("Background job not active, returning");
                return;
            }

            TrustBasketStatusResponse trustBasketStatusResponse =
                    trustClientProvider.getTrustClientForPaymentProfile(invoice.getPaymentProfile()).getBasketStatus(
                            invoice.getPurchaseToken(), TrustUserInfoHelper.createUserInfo(invoice)
                    );

            if (trustBasketStatusResponse.getClearTs() != null) {
                TTrustPaymentCleared.Builder eventBuilder = TTrustPaymentCleared.newBuilder();
                eventBuilder.setClearedAt(ProtoUtils.fromInstant(trustBasketStatusResponse.getClearTs()));
                if (trustBasketStatusResponse.getClearRealTs() != null) {
                    eventBuilder.setRealClearedAt(ProtoUtils.fromInstant(trustBasketStatusResponse.getClearRealTs()));
                }
                workflowMessageSender.scheduleEvent(
                        invoice.getWorkflow().getId(),
                        eventBuilder.build()
                );
                invoice.setBackgroundJobActive(false);
                invoice.setNextCheckStatusAt(null);
            } else {
                invoice.rescheduleNextCheckStatusAt(trustHotelsProperties.getClearingRefreshTimeout());
            }
        } catch (Exception e) {
            log.error("Exception getting trust status clearing", e);
            invoice.rescheduleNextCheckStatusAt(trustHotelsProperties.getClearingRefreshTimeout());
        }
    }
}
