package ru.yandex.travel.orders.workflows.notification.handlers;

import java.time.Instant;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.WellKnownWorkflowEntityType;
import ru.yandex.travel.orders.entities.notifications.Attachment;
import ru.yandex.travel.orders.entities.notifications.Notification;
import ru.yandex.travel.orders.repository.AttachmentRepository;
import ru.yandex.travel.orders.workflow.notification.proto.EAttachmentState;
import ru.yandex.travel.orders.workflow.notification.proto.ENotificationState;
import ru.yandex.travel.orders.workflow.notification.proto.TAttachmentFetched;
import ru.yandex.travel.orders.workflow.notification.proto.TPreparingExpired;
import ru.yandex.travel.orders.workflow.notification.proto.TSendingStart;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.TWorkflowCrashed;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class PreparingAttachmentsStateHandler extends AnnotatedStatefulWorkflowEventHandler<ENotificationState, Notification> {
    private final AttachmentRepository attachmentRepository;

    @HandleEvent
    public void handleAttachmentFetched(TAttachmentFetched event, StateContext<ENotificationState, Notification> context) {
        checkAttachmentAndSend(context, isExpired(context));
    }

    @HandleEvent
    public void handleCheckAttachments(TPreparingExpired event, StateContext<ENotificationState, Notification> context) {
        checkAttachmentAndSend(context, isExpired(context));
    }

    @HandleEvent
    public void handleWorkflowCrashed(TWorkflowCrashed event, StateContext<ENotificationState, Notification> context) {
        checkTWorkflowCrashed(event, attachmentRepository);
        checkAttachmentAndSend(context, isExpired(context));
    }

    private static boolean isExpired(StateContext<ENotificationState, Notification> context) {
        if (context.getWorkflowEntity().getPreparingAttachmentsTill() == null) {
            return false;
        }
        return context.getWorkflowEntity().getPreparingAttachmentsTill().isBefore(Instant.now());
    }

    public static void checkTWorkflowCrashed(TWorkflowCrashed event, AttachmentRepository attachmentRepository) {
        if (!WellKnownWorkflowEntityType.ATTACHMENT.equalsValue(event.getEntityType())) {
            throw new RuntimeException(String.format("Can't handle TWorkflowCrashed with EntityType %s", event.getEntityType()));
        }
        var attachment = attachmentRepository.getOne(UUID.fromString(event.getEntityId()));
        if (attachment.isRequired()) {
            throw new RuntimeException("Required attachment workflow is crashed: " + attachment.getId());
        }
    }

    public static void checkAttachmentAndSend(StateContext<ENotificationState, Notification> context, boolean expired) {
        Notification notification = context.getWorkflowEntity();

        var requiredFetched = notification.getAttachments().stream().filter(Attachment::isRequired)
                .allMatch(x -> x.getState() == EAttachmentState.AS_FETCHED);
        var minorFetched = notification.getAttachments().stream().filter(x -> !x.isRequired())
                .allMatch(x -> x.getState() == EAttachmentState.AS_FETCHED || x.getWorkflow().getState() == EWorkflowState.WS_CRASHED);

        if (expired && !requiredFetched) {
            context.setState(ENotificationState.NS_PREPARING_REQUIRED_ATTACHMENTS);
        } else if (requiredFetched && (expired || minorFetched)) {
            // TODO(ganintsev): if expired and !minorFetched we can send CANCEL message to attachments in NEW state
            context.setState(ENotificationState.NS_SENDING);
            context.scheduleEvent(TSendingStart.getDefaultInstance());
        }
    }
}
