package ru.yandex.travel.orders.workflows.notification.handlers;

import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.io.BaseEncoding;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.bolts.internal.NotImplementedException;
import ru.yandex.travel.orders.entities.notifications.Attachment;
import ru.yandex.travel.orders.entities.notifications.AttachmentProviderType;
import ru.yandex.travel.orders.entities.notifications.EmailChannelInfo;
import ru.yandex.travel.orders.entities.notifications.MailSenderAttachment;
import ru.yandex.travel.orders.entities.notifications.MailTarget;
import ru.yandex.travel.orders.entities.notifications.Notification;
import ru.yandex.travel.orders.entities.notifications.NotificationFailureReason;
import ru.yandex.travel.orders.entities.notifications.SmsChannelInfo;
import ru.yandex.travel.orders.entities.notifications.TemplatedEmailChannelInfo;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.services.MailSenderService;
import ru.yandex.travel.orders.services.YaSmsService;
import ru.yandex.travel.orders.services.YaSmsServiceException;
import ru.yandex.travel.orders.services.cloud.s3.S3Service;
import ru.yandex.travel.orders.services.notifications.NotificationMeters;
import ru.yandex.travel.orders.services.notifications.TemplatedMailSenderService;
import ru.yandex.travel.orders.workflow.notification.proto.ENotificationState;
import ru.yandex.travel.orders.workflow.notification.proto.TAttachmentFetched;
import ru.yandex.travel.orders.workflow.notification.proto.TPreparingExpired;
import ru.yandex.travel.orders.workflow.notification.proto.TSendingStart;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.TWorkflowCrashed;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@RequiredArgsConstructor
@IgnoreEvents(types = {TAttachmentFetched.class, TPreparingExpired.class, TWorkflowCrashed.class})
public class SendingStateHandler extends AnnotatedStatefulWorkflowEventHandler<ENotificationState, Notification> {

    private final MailSenderService mailSenderService;
    private final TemplatedMailSenderService templatedMailSenderService;
    private final YaSmsService smsSenderService;
    private final StarTrekService starTrekService;
    private final S3Service s3Service;
    private final NotificationMeters notificationMeters;

    @HandleEvent
    public void handleSendingStart(TSendingStart event, StateContext<ENotificationState, Notification> context) {
        Notification notification = context.getWorkflowEntity();
        var channelInfo = notification.getChannelInfo();
        try {
            switch (notification.getChannel()) {
                case TEMPLATED_EMAIL:
                    ensureMinorFiscalReceiptFetched(notification, context);
                    sendTemplatedEmailNotification(notification, (TemplatedEmailChannelInfo) channelInfo);
                    break;
                case EMAIL:
                    ensureMinorFiscalReceiptFetched(notification, context);
                    sendEmailNotification(notification, (EmailChannelInfo) channelInfo);
                    break;
                case SMS:
                    var smsChannelInfo = (SmsChannelInfo) channelInfo;
                    smsSenderService.sendSms(smsChannelInfo.getText(), smsChannelInfo.getPhone());
                    break;
                default:
                    throw new NotImplementedException(String.format("Sending to channel %s is not implemented",
                            notification.getChannel()));
            }
            notificationMeters.getNotificationsSent().get(notification.getChannel()).increment();
            context.setState(ENotificationState.NS_SENT);
        } catch (YaSmsServiceException ex) {
            // fail without ST-issue
            if (ex.isBadPhoneError()) {
                notificationMeters.getNotificationsFailed()
                        .get(notification.getChannel())
                        .get(NotificationFailureReason.BAD_PHONE).increment();
                context.setState(ENotificationState.NS_FAILED);
            } else if (ex.isLimitExceededError()) {
                notificationMeters.getNotificationsFailed()
                        .get(notification.getChannel())
                        .get(NotificationFailureReason.LIMIT_EXCEEDED).increment();
                context.setState(ENotificationState.NS_FAILED);
            } else if (ex.isTooLargeError()) {
                notificationMeters.getNotificationsFailed()
                        .get(notification.getChannel())
                        .get(NotificationFailureReason.MESSAGE_TOO_LARGE).increment();
                context.setState(ENotificationState.NS_FAILED);
            } else {
                throw ex;
            }
        }
    }

    private void sendEmailNotification(Notification notification, EmailChannelInfo channelInfo) {
        List<MailSenderAttachment> attachments = notification.getAttachments().stream()
                .filter(SendingStateHandler::hasData)
                .map(attachment -> new MailSenderAttachment(attachment.getFilename(), attachment.getMimeType(),
                        encodeAttachmentData(getData(attachment))))
                .collect(Collectors.toList());
        List<MailTarget> multipleTargets = channelInfo.getMultipleTargets() == null ? null :
                channelInfo.getMultipleTargets().stream()
                        .map(MailTarget::new)
                        .collect(Collectors.toList());
        List<MailTarget> copyTargets = channelInfo.getCopyTargets() == null ? null :
                channelInfo.getCopyTargets().stream()
                        .map(MailTarget::new)
                        .collect(Collectors.toList());
        List<MailTarget> bccTargets = channelInfo.getBccTargets() == null ? null :
                channelInfo.getBccTargets().stream()
                        .map(MailTarget::new)
                        .collect(Collectors.toList());
        mailSenderService.sendEmailSync(
                channelInfo.getCampaign(),
                channelInfo.getTarget(),
                multipleTargets,
                copyTargets,
                bccTargets,
                channelInfo.getArguments(),
                attachments);
    }

    public void sendTemplatedEmailNotification(Notification notification, TemplatedEmailChannelInfo channelInfo) {
        List<MailSenderAttachment> attachments = notification.getAttachments().stream()
                .filter(SendingStateHandler::hasData)
                .map(attachment -> new MailSenderAttachment(attachment.getFilename(), attachment.getMimeType(),
                        encodeAttachmentData(getData(attachment))))
                .collect(Collectors.toList());
        templatedMailSenderService.sendEmailSync(
                channelInfo.getTarget(),
                channelInfo.getTemplate(),
                channelInfo.getArguments(),
                attachments);
    }

    private void ensureMinorFiscalReceiptFetched(
            Notification notification, StateContext<ENotificationState, Notification> context) {
        List<UUID> failedReceiptIds = notification.getAttachments().stream()
                .filter(a -> a.getProvider() == AttachmentProviderType.FISCAL_RECEIPT && !a.isRequired()
                        && !hasData(a))
                .map(Attachment::getId)
                .collect(Collectors.toList());

        if (!failedReceiptIds.isEmpty()) {
            starTrekService.createIssueForTrustReceiptNotFetched(
                    notification.getOrderId(), notification.getOrderPrettyId(), failedReceiptIds, context);
        }
    }

    private static boolean hasData(Attachment attachment) {
        return attachment.getDataS3Id() != null;
    }

    private byte[] getData(Attachment attachment) {
        return s3Service.readObject(attachment.getDataS3Id()).getData();
    }

    private String encodeAttachmentData(byte[] data) {
        return BaseEncoding.base64().encode(data);
    }
}
