package ru.yandex.travel.orders.workflows.notification.jobs;


import java.time.Instant;
import java.util.List;
import java.util.Set;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.data.domain.PageRequest;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.notifications.Notification;
import ru.yandex.travel.orders.repository.NotificationRepository;
import ru.yandex.travel.orders.workflow.notification.proto.ENotificationState;
import ru.yandex.travel.orders.workflow.notification.proto.TPreparingExpired;
import ru.yandex.travel.tx.utils.TransactionMandatory;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.WorkflowProcessService;

@Service
@Slf4j
@RequiredArgsConstructor
public class NotificationRefreshService {
    private final NotificationRepository notificationRepository;

    private final WorkflowProcessService workflowProcessService;

    @TransactionMandatory
    public List<UUID> fetchNotificationsPreparingExpired(Set<UUID> active, int maxResultSize) {
        return notificationRepository.getNotificationsPreparingTill(Instant.now(),
                ENotificationState.NS_PREPARING_ATTACHMENTS, EWorkflowState.WS_RUNNING, false, active,
                PageRequest.of(0, maxResultSize));
    }

    @TransactionMandatory
    public long getNotificationsPreparingExpiredCount(Set<UUID> active) {
        return notificationRepository.countNotificationsPreparingTill(Instant.now(),
                ENotificationState.NS_PREPARING_ATTACHMENTS, EWorkflowState.WS_RUNNING, false, active);
    }

    @TransactionMandatory
    public void handleNotificationPreparingExpired(UUID orderItemId) {
        Notification notification = notificationRepository.getOne(orderItemId);
        if (notification.getState() == ENotificationState.NS_PREPARING_ATTACHMENTS && !notification.isExpired()) {
            notification.setExpired(true);
            workflowProcessService.scheduleEvent(notification.getWorkflow().getId(),
                    TPreparingExpired.getDefaultInstance());
        }
    }
}
