package ru.yandex.travel.orders.workflows.notification.jobs;

import java.util.Collection;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.transaction.TransactionDefinition;
import org.springframework.transaction.support.DefaultTransactionDefinition;

import ru.yandex.travel.orders.workflows.notification.NotificationWorkflowProperties;
import ru.yandex.travel.spring.tx.ForcedRollbackTxManagerWrapper;
import ru.yandex.travel.task_processor.AbstractTaskKeyProvider;
import ru.yandex.travel.task_processor.TaskKeyProvider;
import ru.yandex.travel.task_processor.TaskProcessor;
import ru.yandex.travel.task_processor.TaskProcessorProperties;

@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties(NotificationWorkflowProperties.class)
public class NotificationTaskProcessorsConfiguration {

    private final NotificationRefreshService notificationRefreshService;

    private final ForcedRollbackTxManagerWrapper forcedRollbackTxManagerWrapper;

    private final NotificationWorkflowProperties notificationWorkflowProperties;

    @Bean
    public TaskProcessor<UUID> notificationsPreparingExpiredProcessor() {

        TaskKeyProvider<UUID> taskKeyProvider = new AbstractTaskKeyProvider<>() {
            @Override
            public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
                return notificationRefreshService.fetchNotificationsPreparingExpired(getLockedTaskKeys(),
                        maxResultSize);
            }

            @Override
            public long getPendingTasksCount() {
                return notificationRefreshService.getNotificationsPreparingExpiredCount(getLockedTaskKeys());
            }
        };
        DefaultTransactionDefinition transactionDefinition = new DefaultTransactionDefinition();
        transactionDefinition.setName("NotificationsPreparingExpired");
        transactionDefinition.setPropagationBehavior(TransactionDefinition.PROPAGATION_REQUIRES_NEW);
        transactionDefinition.setIsolationLevel(TransactionDefinition.ISOLATION_REPEATABLE_READ);
        TransactionDefinition txDefinition = new DefaultTransactionDefinition(transactionDefinition);

        return new TaskProcessor<>(taskKeyProvider, notificationRefreshService::handleNotificationPreparingExpired,
                forcedRollbackTxManagerWrapper, txDefinition,
                TaskProcessorProperties.builder().name("NotificationsPreparingExpired")
                        .daemonPoolThreads(true).gracefulPoolShutdown(true)
                        .poolSize(notificationWorkflowProperties.getPreparingExpiredTaskNumberOfItems())
                        .initialStartDelay(notificationWorkflowProperties.getPreparingExpiredTaskInitialStartDelay())
                        .scheduleRate(notificationWorkflowProperties.getPreparingExpiredTaskPeriod()).build());
    }
}
