package ru.yandex.travel.orders.workflows.order;

import java.util.List;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;

import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.GenericOrderUserRefund;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.TrainOrderOfficeRefund;
import ru.yandex.travel.orders.entities.TrainOrderUserRefund;
import ru.yandex.travel.orders.entities.TrainTicketRefund;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.workflow.orderitem.train.ticketrefund.proto.ETrainTicketRefundState;

public class OrderUtils {

    private OrderUtils() {
    }

    public static TrustInvoice getRequiredCurrentInvoice(Order order) {
        Invoice invoice = order.getCurrentInvoice();
        Preconditions.checkState(invoice != null, "Current invoice must be present for order");
        return (TrustInvoice) invoice;
    }

    public static List<TrainTicketRefund> getTrainTicketRefunds(OrderRefund refund) {
        // TODO(ganintsev, tlg-13): remove this type check (TRAVELBACK-1668)
        if (refund instanceof GenericOrderUserRefund) {
            var genericRefund = (GenericOrderUserRefund) refund;
            return genericRefund.getTrainTicketRefunds();
        } else if (refund instanceof TrainOrderUserRefund) {
            var trainRefund = (TrainOrderUserRefund) refund;
            return trainRefund.getTrainTicketRefunds();
        } else if (refund instanceof TrainOrderOfficeRefund) {
            var trainRefund = (TrainOrderOfficeRefund) refund;
            return trainRefund.getTrainTicketRefunds();
        } else {
            return List.of();
        }
    }

    public static List<TrainTicketRefund> getSuccessfulTicketRefunds(OrderRefund refund) {
        return getTrainTicketRefunds(refund).stream()
                .filter(x -> x.getState() == ETrainTicketRefundState.RS_REFUNDED)
                .collect(Collectors.toList());
    }

    public static EDisplayOrderType mapServiceTypeToDisplayOrderType(EServiceType serviceType) {
        switch (serviceType) {
            case PT_BNOVO_HOTEL:
            case PT_DOLPHIN_HOTEL:
            case PT_EXPEDIA_HOTEL:
            case PT_BRONEVIK_HOTEL:
            case PT_TRAVELLINE_HOTEL:
                return EDisplayOrderType.DT_HOTEL;
            case PT_BUS:
                return EDisplayOrderType.DT_BUS;
            case PT_FLIGHT:
                return EDisplayOrderType.DT_AVIA;
            case PT_TRAIN:
                return EDisplayOrderType.DT_TRAIN;
            case PT_SUBURBAN:
                return EDisplayOrderType.DT_SUBURBAN;
            default:
                throw new IllegalArgumentException("Unknown service type value: " + serviceType);
        }
    }
}
