package ru.yandex.travel.orders.workflows.order.aeroflot;

import java.time.Clock;
import java.time.Instant;
import java.time.LocalDate;
import java.time.LocalDateTime;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.entities.AeroflotOrder;
import ru.yandex.travel.orders.entities.promo.DiscountApplicationConfig;
import ru.yandex.travel.orders.entities.promo.OrderGeneratedPromoCodes;
import ru.yandex.travel.orders.entities.promo.PromoCode;
import ru.yandex.travel.orders.entities.promo.PromoCodeHelpers;
import ru.yandex.travel.orders.repository.promo.PromoCodeRepository;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.services.OperationTypes;
import ru.yandex.travel.orders.services.TimeHelpers;
import ru.yandex.travel.orders.services.promo.PromoCodeGenerationService;
import ru.yandex.travel.orders.services.promo.PromoCodeUnifier;
import ru.yandex.travel.orders.services.promo.simple.PromoSendMailData;
import ru.yandex.travel.orders.workflows.orderitem.aeroflot.configuration.AeroflotWorkflowProperties;
import ru.yandex.travel.workflow.single_operation.SingleOperationService;

@Service
@RequiredArgsConstructor
@Slf4j
public class AeroflotWorkflowService {
    private final AeroflotWorkflowProperties aeroflotWorkflowProperties;
    private final SingleOperationService singleOperationService;
    private final PromoCodeRepository promoCodeRepository;
    private final PromoCodeGenerationService promoCodeGenerationService;
    private final Clock clock;

    public void issueHotelPromoCodeOnConfirmation(AeroflotOrder order) {
        createPromoCodeForHotels(order);
        scheduleAeroflotPromoMail(order);
    }

    private void scheduleAeroflotPromoMail(AeroflotOrder aviaOrder) {
        if (!aeroflotWorkflowProperties.getPromo().getEnabled()) {
            return;
        }
        var stationToSettlementGeoId = getStationToSettlementGeoId(aviaOrder);
        var dataBuilder = PromoSendMailData.builder()
                .parentEntityId(aviaOrder.getId())
                .parentEntityType(aviaOrder.getEntityType())
                .promoCampaignName("AviaHotel2021")
                .emailCampaign(aeroflotWorkflowProperties.getPromo().getCampaign())
                .email(aviaOrder.getEmail())
                .stationToSettlementGeoId(stationToSettlementGeoId);
        PromoCode promoCode = null;
        if (aeroflotWorkflowProperties.getPromo().isGeneratePersonalPromoCodes() &&
                aviaOrder.getGeneratedPromoCodes() != null &&
                aviaOrder.getGeneratedPromoCodes().getPromoCodes() != null &&
                aviaOrder.getGeneratedPromoCodes().getPromoCodes().size() > 0) {
            promoCode = aviaOrder.getGeneratedPromoCodes().getPromoCodes().get(0);
        } else if (!Strings.isNullOrEmpty(aeroflotWorkflowProperties.getPromo().getCommonPromoCodeForHotels())){
            String code = PromoCodeUnifier.unifyCode(aeroflotWorkflowProperties.getPromo().getCommonPromoCodeForHotels());
            promoCode = promoCodeRepository.findByCodeEquals(code);
        } else {
            log.error("No promo codes for avia promo mail found. Either generate one-time-promo-code or fill valid common-promo-code in config");
        }
        if (promoCode != null) {
            DiscountApplicationConfig discountApplicationConfig =
                    promoCode.getPromoAction().getDiscountApplicationConfig();
            LocalDateTime validTill = TimeHelpers.toLocalDateTime(PromoCodeHelpers.getPromoCodeValidTill(promoCode),
                    TimeHelpers.MSK_TZ);
            dataBuilder.promoCode(promoCode.getCode())
                    .codeAddsUpWithOtherActions(discountApplicationConfig.isAddsUpWithOtherActions())
                    .codeNominalType(promoCode.getNominalType().toString())
                    .codeNominal(promoCode.getNominal().intValueExact())
                    .codeMinTotalCost(discountApplicationConfig.getMinTotalCost().getNumber().intValueExact())
                    .codeValidTill(NotificationHelper.humanDate(validTill));
            singleOperationService.scheduleOperation(
                    "AviaHotel2021Promo" + Instant.now(clock).toEpochMilli(),
                    OperationTypes.SIMPLE_PROMO_EMAIL_SENDER.getValue(),
                    dataBuilder.build(),
                    Instant.now(clock).plus(aeroflotWorkflowProperties.getPromo().getDelay())
            );
        }
    }

    private Integer getStationToSettlementGeoId(AeroflotOrder aviaOrder) {
        var aviaOrderPayload = aviaOrder
                .getAeroflotOrderItem()
                .getPayload();
        var arrivalStationCode = aviaOrderPayload
                .getVariant()
                .getOriginDestinations()
                .get(0)
                .getArrivalCode();
        var stationToSettlement = aviaOrderPayload
                .getAviaDicts()
                .getSettlements()
                .stream()
                .filter(settlement -> settlement.getCode().equals(arrivalStationCode))
                .findFirst();
        return stationToSettlement.isPresent() ? stationToSettlement.get().getGeoID() : null;
    }

    private void createPromoCodeForHotels(AeroflotOrder order) {
        if (!aeroflotWorkflowProperties.getPromo().isGeneratePersonalPromoCodes()) {
            return;
        }
        if (Strings.isNullOrEmpty(aeroflotWorkflowProperties.getPromo().getPromoActionForHotels())) {
            log.error("Promo action for avia-for-hotels promoCode not set. Please check config. Order: {}", order.getId());
            return;
        }

        String actionName = aeroflotWorkflowProperties.getPromo().getPromoActionForHotels();
        PromoCode promoCode = promoCodeGenerationService.generatePromoCodeForAction(actionName, LocalDate.now());
        if (promoCode != null) {
            if (order.getGeneratedPromoCodes() == null) {
                OrderGeneratedPromoCodes.createForOrder(order);
            }
            order.getGeneratedPromoCodes().addPromoCode(promoCode);
            log.info("Created promo code {} for order {} using action {}", promoCode.getCode(), order.getId(), actionName);
        } else {
            log.warn("Could not create promo code for order {} using action {}", order.getId(), actionName);
        }
    }
}
