package ru.yandex.travel.orders.workflows.order.aeroflot;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import com.google.common.base.Preconditions;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotTicketCoupon;
import ru.yandex.avia.booking.partners.gateways.model.booking.TravellerInfo;
import ru.yandex.travel.orders.entities.AeroflotOrderItem;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotOrderCouponStateMapper;
import ru.yandex.travel.orders.workflow.orderitem.aeroflot.proto.TAeroflotOrderItemTicketCoupons;

@Slf4j
public class AeroflotWorkflowUtils {
    public static AeroflotOrderItem getOnlyOrderItem(Order order) {
        List<OrderItem> items = order.getOrderItems();
        Preconditions.checkArgument(items.size() == 1, "exactly 1 order item is expected: items=%s", items);
        OrderItem orderItem = items.get(0);
        Preconditions.checkArgument(orderItem instanceof AeroflotOrderItem,
                "expected orderItem of type AeroflotOrderItem but have got %s", orderItem.getClass().getName());
        return (AeroflotOrderItem) orderItem;
    }

    public static Invoice getOnlyInvoice(Order order) {
        List<Invoice> items = order.getInvoices();
        Preconditions.checkArgument(items.size() == 1, "exactly 1 invoice is expected: items=%s", items);
        return items.get(0);
    }

    public static String composeAviaPnr(String pnr, String pnrDate) {
        return pnr + "/" + pnrDate;
    }

    public static void updateTickets(AeroflotServicePayload payload, List<String> tickets) {
        List<TravellerInfo> travellers = payload.getTravellers();
        if (travellers.size() != tickets.size()) {
            log.warn("Travellers/Tickets mismatch; ignoring the sync event; travellers={}, tickets={}", travellers, tickets);
            return;
        }

        Map<String, List<String>> ticketsMapping = new HashMap<>();
        for (int i = 0; i < travellers.size(); i++) {
            ticketsMapping.put(travellers.get(i).getTravellerInfoId(), List.of(tickets.get(i)));
        }
        if (payload.getTickets() == null) {
            payload.setTickets(ticketsMapping);
        } else {
            // the event may be processed multiple times but the tickets should always be the same
            Preconditions.checkArgument(payload.getTickets().equals(ticketsMapping),
                    "Trying to re-assign different tickets to order item; current=%s, new=%s",
                    payload.getTickets(), ticketsMapping);
        }
    }

    public static void updateTicketCoupons(AeroflotServicePayload payload, Map<String, TAeroflotOrderItemTicketCoupons> ticketsCoupons) {
        if(ticketsCoupons != null) {
            Map<String, List<AeroflotTicketCoupon>> ticketCoupons = new HashMap<>();
            for (var coupons : ticketsCoupons.entrySet()) {
                var tcoupons = new ArrayList<AeroflotTicketCoupon>();
                for (var c : coupons.getValue().getCouponsMap().entrySet()) {
                    tcoupons.add(new AeroflotTicketCoupon(c.getKey(), AeroflotOrderCouponStateMapper.getCouponStatusCode(c.getValue())));
                }
                ticketCoupons.put(coupons.getKey(), tcoupons);
            }

            payload.setTicketCoupons(ticketCoupons);
        }
    }
}
