package ru.yandex.travel.orders.workflows.order.aeroflot.jobs;

import java.time.Instant;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.Value;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.commons.logging.NestedMdc;
import ru.yandex.travel.orders.repository.AeroflotOrderRepository;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotOrderStateSync;
import ru.yandex.travel.task_processor.AbstractSimpleTask;

@Service
@RequiredArgsConstructor
@Slf4j
public class AeroflotOrderStateSyncTask extends AbstractSimpleTask<UUID> {
    private final AeroflotOrderRepository aeroflotOrderRepository;
    private final AeroflotOrderStateSync singleOrderSyncer;
    public final List<OrderRefreshSchedule> ordersToRefresh = Collections.synchronizedList(new ArrayList<>());

    @Value
    public static class OrderRefreshSchedule {
        UUID uuid;
        Instant at;
    }

    @Override
    public void process(UUID orderId) {
        log.info("Start syncing order state {}", orderId);
        try {
            var optOrder = aeroflotOrderRepository.findById(orderId);
            if (optOrder.isEmpty()) {
                log.error("Order {} was queued for update, but not found", orderId);
                return;
            }
            var order = optOrder.get();
            try (var ignored = NestedMdc.forEntity(order)) {
                log.info("Checking aeroflot order state");
                try {
                    singleOrderSyncer.sync(order);
                } catch (RuntimeException e) {
                    log.error("Error updating aeroflot order state {}", order.getId(), e);
                }
                log.info("Checked aeroflot order state");
            }
        } finally {
            log.info("Done syncing order state {}", orderId);
            ordersToRefresh.removeIf(orderRefreshSchedule -> orderRefreshSchedule.uuid == orderId);
        }
    }

    @Override
    public Collection<UUID> getPendingTaskKeys(int maxResultSize) {
        return ordersToRefresh.stream()
                .filter(orderRefreshSchedule -> orderRefreshSchedule.at.isBefore(Instant.now()))
                .map(orderRefreshSchedule -> orderRefreshSchedule.uuid)
                .collect(Collectors.toList());
    }

    @Override
    public long getPendingTasksCount() {
        return ordersToRefresh.stream()
                .filter(orderRefreshSchedule -> orderRefreshSchedule.at.isBefore(Instant.now()))
                .count();
    }
}
