package ru.yandex.travel.orders.workflows.order.aeroflot.jobs;

import java.time.Instant;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.List;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.repository.AeroflotOrderRepository;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotOrderStateSyncLimit;
import ru.yandex.travel.orders.services.avia.aeroflot.AeroflotOrderStateSyncProperties;

@Service
@RequiredArgsConstructor
@Slf4j
@EnableConfigurationProperties(AeroflotOrderStateSyncProperties.class)
public class AeroflotOrdersStateSyncListRefresherTask {
    public static final String TASK_KEY = "AEROFLOT_ORDER_STATE_SYNC";
    private final AeroflotOrderRepository aeroflotOrderRepository;
    private final AeroflotOrderStateSyncLimit rateLimiter;
    private final AeroflotOrderStateSyncTask aeroflotOrderStateSyncTask;

    public void refreshOrdersList(String taskKeyIgnored) {
        log.info("Start aeroflot orders state sync list refresher");
        if (!aeroflotOrderStateSyncTask.ordersToRefresh.isEmpty()) {
            log.info("Orders queue is not empty. Refill it later");
            return;
        }
        var orders = aeroflotOrderRepository.findAllActiveOrders(LocalDateTime.now());
        if (orders.isEmpty()) {
            log.info("No active orders to add to queue");
            return;
        }
        List<AeroflotOrderStateSyncTask.OrderRefreshSchedule> ordersRefreshAtQueue = new ArrayList<>(orders.size());
        var startAt = Instant.now();
        log.info("Going to start refresh at {}", startAt);
        var refreshInterval = rateLimiter.estimateIntervalBetweenOrderUpdates(orders.size());
        log.info("Calculated refresh interval is {}", refreshInterval);
        for (var i = 0; i < orders.size(); i++) {
            var order = orders.get(i);
            var runAt = startAt.plus(refreshInterval.multipliedBy(i));
            ordersRefreshAtQueue.add(new AeroflotOrderStateSyncTask.OrderRefreshSchedule(order.getId(), runAt));
            log.debug("Example: order {} will be refreshed at {}", order, runAt);
        }
        aeroflotOrderStateSyncTask.ordersToRefresh.addAll(ordersRefreshAtQueue);
        log.info("Added {} orders to queue", orders.size());
    }
}
