package ru.yandex.travel.orders.workflows.order.generic;

import java.time.Clock;
import java.time.Instant;
import java.util.List;
import java.util.UUID;

import com.google.common.base.Preconditions;
import com.google.common.io.BaseEncoding;
import com.google.protobuf.InvalidProtocolBufferException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.travel.orders.commons.proto.ECancellationReason;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.GenericOrderUserRefund;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.ServiceRefundState;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.context.OrderItemContextState;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.services.buses.BusNotificationHelper;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.promo.PromoCodeApplicationService;
import ru.yandex.travel.orders.services.train.RebookingService;
import ru.yandex.travel.orders.workflow.notification.proto.TSend;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.order.generic.proto.TGenericRefundToken;
import ru.yandex.travel.orders.workflow.order.generic.proto.TServiceRefundInfo;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TStartReservationCancellation;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.TCancellationStart;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.TReservationStart;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TRefundStart;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherCreated;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherRecreated;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.workflow.StateContext;

@Service
@RequiredArgsConstructor
@Slf4j
public class GenericWorkflowService {
    private final PromoCodeApplicationService promoCodeApplicationService;
    private final RebookingService trainRebookingService;
    private final NotificationHelper notificationHelper;
    private final BusNotificationHelper busNotificationHelper;
    private final Clock clock;

    public void handleServiceCancelled(TServiceCancelled event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        UUID serviceId = UUID.fromString(event.getServiceId());
        order.getStateContext().serviceCancelled(serviceId);
        OrderItem cancelledOrderItem = order.findOrderItemById(serviceId).get();
        if (order.isTrainRebookingEnabled() &&
                order.getExpiresAt().isAfter(Instant.now(clock)) &&
                cancelledOrderItem.isRebookingAllowed()) {
            if (cancelledOrderItem.getPublicType() == EServiceType.PT_TRAIN) {
                OrderItem rebookedItem = trainRebookingService.rebookItem(order,
                        (TrainOrderItem) cancelledOrderItem);
                UUID orderItemWorkflowId = rebookedItem.getWorkflow().getId();
                var itemState = order.getStateContext().getItem(rebookedItem.getId());
                itemState.setState(EOrderItemState.IS_RESERVING);
                itemState.setWorkflowId(orderItemWorkflowId);
                ctx.scheduleExternalEvent(orderItemWorkflowId, TReservationStart.newBuilder().build());
                ctx.setState(EOrderState.OS_WAITING_RESERVATION);
            } else {
                throw new UnsupportedOperationException(String.format("Rebooking items of type %s is not supported",
                        cancelledOrderItem.getPublicType()));
            }
            return;
        }
        if (order.getStateContext().getCancellationReason() == null) {
            if (cancelledOrderItem.isExpired()) {
                order.getStateContext().setCancellationReason(ECancellationReason.CR_EXPIRED);
            } else {
                order.getStateContext().setCancellationReason(ECancellationReason.CR_RESERVATION_FAILED);
            }
        }
        if (order.getStateContext().allItemsCancelled() && !order.getStateContext().isMoneyAcquired()) {
            ctx.setState(EOrderState.OS_CANCELLED);
            promoCodeApplicationService.freePromoCodeActivations(order);
        } else {
            ctx.setState(EOrderState.OS_WAITING_CANCELLATION);
            for (var itemState : order.getStateContext().getOrderItems()) {
                if (itemState.getState() == EOrderItemState.IS_RESERVED) {
                    itemState.setState(EOrderItemState.IS_CANCELLING);
                    ctx.scheduleExternalEvent(itemState.getWorkflowId(), TCancellationStart.getDefaultInstance());
                }
            }
            if (order.getStateContext().isMoneyAcquired()) {
                Invoice invoice = order.getCurrentInvoice();
                ctx.scheduleExternalEvent(invoice.getWorkflow().getId(), InvoiceUtils.buildFullRefund(invoice));
            }
        }
    }

    public void handleStartReservationCancellation(TStartReservationCancellation event,
                                                   StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        order.setUserActionScheduled(false);
        order.getStateContext().setCancellationReason(ECancellationReason.CR_USER_CANCELLED);
        for (OrderItemContextState itemState : order.getStateContext().getOrderItems()) {
            itemState.setState(EOrderItemState.IS_CANCELLING);
            ctx.scheduleExternalEvent(itemState.getWorkflowId(), TCancellationStart.getDefaultInstance());
        }
        ctx.setState(EOrderState.OS_WAITING_CANCELLATION);
    }

    public void scheduleServiceRefund(GenericOrder order, GenericOrderUserRefund refund,
                                      TGenericRefundToken token, UUID serviceIdToRefund,
                                      StateContext<EOrderState, GenericOrder> context) {
        List<TServiceRefundInfo> services = token.getServiceList();
        TServiceRefundInfo serviceRefundInfo = services.stream()
                .filter(s -> UUID.fromString(s.getServiceId()).equals(serviceIdToRefund))
                .findFirst().orElseThrow();
        var itemState = order.getStateContext().getItem(serviceIdToRefund);
        Preconditions.checkState(itemState.getState() == EOrderItemState.IS_CONFIRMED,
                "Expected item state CONFIRMED, but actual: %s", itemState.getState());
        itemState.setState(EOrderItemState.IS_REFUNDING);
        refund.getPayload().changeServiceRefundState(serviceIdToRefund, ServiceRefundState.PENDING,
                ServiceRefundState.REFUNDING);
        if (serviceRefundInfo.getOneOfServiceRefundInfoCase() == TServiceRefundInfo.OneOfServiceRefundInfoCase.TRAINREFUNDTOKEN) {
            context.scheduleExternalEvent(itemState.getWorkflowId(), TRefundStart.newBuilder()
                    .setToken(BaseEncoding.base64Url().encode(serviceRefundInfo.getTrainRefundToken().toByteArray()))
                    .setOrderRefundId(refund.getId().toString())
                    .build());
        } else if (serviceRefundInfo.getOneOfServiceRefundInfoCase() == TServiceRefundInfo.OneOfServiceRefundInfoCase.BUSREFUNDTOKEN) {
            context.scheduleExternalEvent(itemState.getWorkflowId(), ru.yandex.travel.orders.workflow.orderitem
                    .bus.proto.TRefundStart.newBuilder()
                    .setToken(serviceRefundInfo.getBusRefundToken())
                    .setOrderRefundId(refund.getId().toString())
                    .build());
        } else {
            throw new UnsupportedOperationException("Refund is not implemented for " + serviceRefundInfo.getOneOfServiceRefundInfoCase());
        }
    }

    public TGenericRefundToken getRefundToken(String token) {
        try {
            return TGenericRefundToken.parseFrom(BaseEncoding.base64Url().decode(token));
        } catch (InvalidProtocolBufferException e) {
            throw new RuntimeException("Unable to deserialize refund token", e);
        }
    }

    public void handleVoucherCreated(TVoucherCreated event, StateContext<EOrderState, GenericOrder> context) {
        GenericOrder order = context.getWorkflowEntity();
        order.setDocumentUrl(event.getVoucherUrl());
        order.getStateContext().setDocumentsUpdatedAt(Instant.now());

        if (OrderCompatibilityUtils.isHotelOrder(order)) {
            UUID notificationWorkflowId = notificationHelper.createWorkflowForSuccessfulHotelNotification(order);
            var senderRequest = TSend.newBuilder().build();
            context.scheduleExternalEvent(notificationWorkflowId, senderRequest);
        } else if (OrderCompatibilityUtils.isBusOrder(order)) {
            var emailWorkflowId = busNotificationHelper.createWorkflowForConfirmedOrderEmail(order);
            var sendEvent = TSend.newBuilder().build();
            context.scheduleExternalEvent(emailWorkflowId, sendEvent);
            try {
                var smsWorkflowId = busNotificationHelper.createWorkflowForConfirmedOrderSms(order);
                context.scheduleExternalEvent(smsWorkflowId, sendEvent);
            } catch (Exception e) {
                log.error("Unable to create workflow for successful bus SMS. SMS won't be sent", e);
            }
        }
    }

    public void handleVoucherRecreated(TVoucherRecreated event, StateContext<EOrderState, GenericOrder> context) {
        GenericOrder order = context.getWorkflowEntity();
        order.setDocumentUrl(event.getVoucherUrl());
        order.getStateContext().setDocumentsUpdatedAt(Instant.now());
    }
}
