package ru.yandex.travel.orders.workflows.order.generic.handlers;

import java.time.Instant;
import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.GenericOrderUserRefund;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderRefundServiceState;
import ru.yandex.travel.orders.entities.ServiceRefundState;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrainOrderOfficeRefund;
import ru.yandex.travel.orders.entities.Voucher;
import ru.yandex.travel.orders.entities.WellKnownWorkflow;
import ru.yandex.travel.orders.entities.context.SimpleMultiItemBatch;
import ru.yandex.travel.orders.repository.OrderRefundRepository;
import ru.yandex.travel.orders.repository.VoucherRepository;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.workflow.notification.proto.TNotificationComplete;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.order.generic.proto.TGenericRefundToken;
import ru.yandex.travel.orders.workflow.order.generic.proto.TServiceRefundInfo;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TManualRefund;
import ru.yandex.travel.orders.workflow.order.proto.TRenderDocuments;
import ru.yandex.travel.orders.workflow.order.proto.TStartServiceRefund;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TChangeRegistrationStatus;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TOfficeRefundOccured;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TUpdateTickets;
import ru.yandex.travel.orders.workflow.train.proto.TOrderOfficeRefundStart;
import ru.yandex.travel.orders.workflow.train.proto.TRegistrationStatusChange;
import ru.yandex.travel.orders.workflow.train.proto.TRegistrationStatusChanged;
import ru.yandex.travel.orders.workflow.train.proto.TServiceOfficeRefundStart;
import ru.yandex.travel.orders.workflow.train.proto.TTrainTicketsUpdated;
import ru.yandex.travel.orders.workflow.train.proto.TUpdateTrainTickets;
import ru.yandex.travel.orders.workflow.voucher.proto.TGenerateVoucher;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherCreated;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherRecreated;
import ru.yandex.travel.orders.workflows.order.generic.GenericWorkflowService;
import ru.yandex.travel.train.model.PassengerCategory;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;
import ru.yandex.travel.workflow.entities.Workflow;
import ru.yandex.travel.workflow.repository.WorkflowRepository;

import static java.util.stream.Collectors.groupingBy;
import static java.util.stream.Collectors.toMap;
import static ru.yandex.travel.orders.entities.context.SimpleMultiItemBatchTaskState.COMPLETED;
import static ru.yandex.travel.orders.entities.context.SimpleMultiItemBatchTaskState.IN_PROGRESS;

@IgnoreEvents(types = TInvoiceCleared.class)
@RequiredArgsConstructor
@Slf4j
public class ConfirmedStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderState, GenericOrder> {
    private final VoucherRepository voucherRepository;
    private final WorkflowRepository workflowRepository;
    private final OrderRefundRepository orderRefundRepository;
    private final GenericWorkflowService genericWorkflowService;

    @HandleEvent
    public void handleRenderDocuments(TRenderDocuments event, StateContext<EOrderState, GenericOrder> context) {
        GenericOrder order = context.getWorkflowEntity();
        Voucher voucher = Voucher.createForOrder(order);
        voucher = voucherRepository.saveAndFlush(voucher);
        Workflow voucherWorkflow = Workflow.createWorkflowForEntity(voucher,
                WellKnownWorkflow.GENERIC_ERROR_SUPERVISOR.getUuid());
        voucherWorkflow = workflowRepository.saveAndFlush(voucherWorkflow);

        context.scheduleExternalEvent(voucherWorkflow.getId(), TGenerateVoucher.newBuilder().build());
    }

    @HandleEvent
    public void handleVoucherCreated(TVoucherCreated event, StateContext<EOrderState, GenericOrder> context) {
        genericWorkflowService.handleVoucherCreated(event, context);
    }

    @HandleEvent
    public void handleVoucherCreated(TVoucherRecreated event, StateContext<EOrderState, GenericOrder> context) {
        genericWorkflowService.handleVoucherRecreated(event, context);
    }

    @HandleEvent
    public void handleNotificationComplete(TNotificationComplete event,
                                           StateContext<EOrderState, GenericOrder> context) {
        log.info("Confirmation notification sent for order {}", context.getWorkflowEntity().getId());
    }

    // Train-Specific Handlers

    @HandleEvent
    public void handle(TUpdateTrainTickets event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(order),
                "Train order expected but got %s", order.getDisplayType());
        SimpleMultiItemBatch tasks = order.getStateContext().getUpdateTicketsTasks();
        tasks.ensureEmpty();
        for (TrainOrderItem orderItem : OrderCompatibilityUtils.getTrainOrderItems(order)) {
            tasks.addTask(orderItem.getId(), IN_PROGRESS);
            ctx.scheduleExternalEvent(orderItem.getWorkflow().getId(), TUpdateTickets.newBuilder().build());
        }
    }

    @HandleEvent
    public void handleTicketsUpdated(TTrainTicketsUpdated event, StateContext<EOrderState, GenericOrder> ctx) {
        Order order = ctx.getWorkflowEntity();
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(order),
                "Train order expected but got %s", order.getDisplayType());
        UUID itemId = UUID.fromString(event.getServiceId());
        SimpleMultiItemBatch tasks = order.getStateContext().getUpdateTicketsTasks();
        tasks.changeTaskState(itemId, IN_PROGRESS, COMPLETED);
        if (tasks.allTasksInState(COMPLETED)) {
            tasks.clear();
            order.toggleUserActionScheduled(false);
        }
        log.info("Service info refreshed for service {}", event.getServiceId());
    }

    @HandleEvent
    public void handleRegistrationStatusChange(TRegistrationStatusChange event,
                                               StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(order),
                "Train order expected but got %s", order.getDisplayType());

        Map<Integer, TrainOrderItem> allBlankIdToItems = OrderCompatibilityUtils.getTrainOrderItems(order).stream()
                .flatMap(oi -> oi.getReservation().getPassengers().stream()
                        // these guys heave the same blank ids as one of the adult passengers
                        .filter(p -> p.getCategory() != PassengerCategory.BABY)
                        .map(p -> Map.entry(oi, p.getTicket().getBlankId())))
                .collect(toMap(Map.Entry::getValue, Map.Entry::getKey));
        Map<TrainOrderItem, List<Integer>> requestedBlankIdsPerItem = event.getBlankIdsList().stream()
                .collect(groupingBy(allBlankIdToItems::get));
        List<Integer> unknownBlankIds = requestedBlankIdsPerItem.get(null);
        Preconditions.checkArgument(unknownBlankIds == null, "Unknown blank ids: %s", unknownBlankIds);

        SimpleMultiItemBatch registrationTasks = order.getStateContext().getTrainRegistrationChangeTasks();
        registrationTasks.ensureEmpty();
        for (TrainOrderItem item : requestedBlankIdsPerItem.keySet()) {
            registrationTasks.addTask(item.getId(), IN_PROGRESS);
            List<Integer> sameItemBlankIds = requestedBlankIdsPerItem.get(item);
            ctx.scheduleExternalEvent(item.getWorkflow().getId(),
                    TChangeRegistrationStatus.newBuilder()
                            .setEnabled(event.getEnabled())
                            .addAllBlankIds(sameItemBlankIds).build());
        }
    }

    @HandleEvent
    public void handleRegistrationStatusChanged(TRegistrationStatusChanged event,
                                                StateContext<EOrderState, GenericOrder> ctx) {
        Order order = ctx.getWorkflowEntity();
        Preconditions.checkArgument(OrderCompatibilityUtils.isTrainOrder(order),
                "Train order expected but got %s", order.getDisplayType());
        UUID itemId = UUID.fromString(event.getServiceId());
        SimpleMultiItemBatch registrationTasksBatch = order.getStateContext().getTrainRegistrationChangeTasks();
        registrationTasksBatch.changeTaskState(itemId, IN_PROGRESS, COMPLETED);
        if (registrationTasksBatch.allTasksInState(COMPLETED)) {
            registrationTasksBatch.clear();
            order.toggleUserActionScheduled(false);
            order.getStateContext().setTrainRegistrationChangedAt(Instant.now());
        }
    }

    @HandleEvent
    public void handleStartRefund(TStartServiceRefund event, StateContext<EOrderState, GenericOrder> context) {
        GenericOrder order = context.getWorkflowEntity();
        TGenericRefundToken token = genericWorkflowService.getRefundToken(event.getToken());
        Preconditions.checkArgument(token.getServiceCount() > 0, "Refund token services must be not empty");
        GenericOrderUserRefund refund = GenericOrderUserRefund.createForOrder(order);
        refund.getPayload().setRefundPartContexts(token.getContextList());
        refund.getPayload().setRefundToken(event.getToken());
        refund.getPayload().setServiceRefundStates(token.getServiceList().stream()
                .map(s -> new OrderRefundServiceState(UUID.fromString(s.getServiceId()), ServiceRefundState.PENDING))
                .collect(Collectors.toList()));

        boolean parallelRefund = true;
        if (token.getServiceList().stream().filter(x -> x.getOneOfServiceRefundInfoCase() ==
                TServiceRefundInfo.OneOfServiceRefundInfoCase.TRAINREFUNDTOKEN).count() > 1) {
            parallelRefund = false;
        }
        order.toggleUserActionScheduled(false);
        order.getStateContext().setRefundsUpdatedAt(Instant.now());
        context.setState(EOrderState.OS_REFUNDING);
        if (parallelRefund) {
            for (var stateItem : refund.getPayload().getServiceRefundStates()) {
                genericWorkflowService.scheduleServiceRefund(order, refund, token, stateItem.getServiceId(), context);
            }
        } else {
            genericWorkflowService.scheduleServiceRefund(order, refund, token,
                    refund.getPayload().nextPendingServiceRefund(), context);
        }
    }

    @HandleEvent
    public void handleOfficeRefundStart(TOrderOfficeRefundStart event,
                                        StateContext<EOrderState, GenericOrder> context) {
        GenericOrder order = context.getWorkflowEntity();
        TrainOrderOfficeRefund refund = TrainOrderOfficeRefund.createForOrder(order);
        refund = orderRefundRepository.save(refund);
        Preconditions.checkArgument(event.getServicesCount() > 0, "Office refund event services must be not empty");
        order.toggleUserActionScheduled(false);
        order.getStateContext().setRefundsUpdatedAt(Instant.now());
        context.setState(EOrderState.OS_REFUNDING);
        for (TServiceOfficeRefundStart serviceRefundMsg : event.getServicesList()) {
            UUID serviceId = UUID.fromString(serviceRefundMsg.getServiceId());
            var itemState = order.getStateContext().getItem(serviceId);
            itemState.setState(EOrderItemState.IS_REFUNDING);
            context.scheduleExternalEvent(itemState.getWorkflowId(), TOfficeRefundOccured.newBuilder()
                    .setOrderRefundId(refund.getId().toString())
                    .addAllRefundOperationIds(serviceRefundMsg.getRefundOperationIdsList()).build());
        }
    }

    @HandleEvent
    public void handleManualMoneyRefund(TManualRefund event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        order.toggleUserActionScheduled(false);
        order.setState(EOrderState.OS_REFUNDING);
        ctx.scheduleEvent(event);
    }
}
