package ru.yandex.travel.orders.workflows.order.generic.handlers;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.context.OrderItemContextState;
import ru.yandex.travel.orders.entities.context.OrderStateContext;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.order.proto.TStartReservation;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.TReservationStart;
import ru.yandex.travel.orders.workflows.order.generic.GenericWorkflowProperties;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@Slf4j
@RequiredArgsConstructor
public class NewStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderState, GenericOrder> {
    private final GenericWorkflowProperties genericWorkflowProperties;

    @HandleEvent
    public void handleStartReservation(TStartReservation event, StateContext<EOrderState, GenericOrder> context) {
        GenericOrder order = context.getWorkflowEntity();
        if (OrderCompatibilityUtils.isTrainOrder(order)) {
            order.setTrainRebookingEnabled(genericWorkflowProperties.isTrainRebookingEnabled() && OrderCompatibilityUtils
                    .getTrainOrderItems(order).stream().noneMatch(x -> x.getPayload().isProviderP2()));
        }
        order.setExpiresAt(order.getCreatedAt().plus(genericWorkflowProperties.getOrderExpirationTime()));

        var stateContext = new OrderStateContext();
        stateContext.init(order.getDisplayType());
        Preconditions.checkState(!order.getOrderItems().isEmpty(), "No order items to reserve");
        for (OrderItem orderItem : order.getOrderItems()) {
            context.scheduleExternalEvent(orderItem.getWorkflow().getId(), TReservationStart.newBuilder().build());
            stateContext.getOrderItems().add(new OrderItemContextState(
                    orderItem.getId(),
                    orderItem.getWorkflow().getId(),
                    EOrderItemState.IS_RESERVING
            ));
        }
        order.setStateContext(stateContext);

        context.setState(EOrderState.OS_WAITING_RESERVATION);
        order.toggleUserActionScheduled(false);
    }
}
