package ru.yandex.travel.orders.workflows.order.generic.handlers;

import java.time.Instant;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.workflow.notification.proto.TNotificationComplete;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TManualRefund;
import ru.yandex.travel.orders.workflow.train.proto.TOrderOfficeRefundStart;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherCreated;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherRecreated;
import ru.yandex.travel.orders.workflows.order.generic.GenericWorkflowService;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@IgnoreEvents(types = {TInvoiceCleared.class, TNotificationComplete.class})
@RequiredArgsConstructor
public class RefundedStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderState, GenericOrder> {
    private final GenericWorkflowService genericWorkflowService;

    @HandleEvent
    public void handleManualMoneyRefund(TManualRefund event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        order.toggleUserActionScheduled(false);
        order.setState(EOrderState.OS_REFUNDING);
        ctx.scheduleEvent(event);
    }

    @HandleEvent
    public void handleOfficeRefundStart(TOrderOfficeRefundStart event, StateContext<EOrderState, GenericOrder> ctx) {
        ctx.getWorkflowEntity().setState(EOrderState.OS_CONFIRMED);
        ctx.scheduleEvent(event);
    }

    @HandleEvent
    public void handleVoucherCreated(TVoucherCreated event, StateContext<EOrderState, GenericOrder> context) {
        GenericOrder order = context.getWorkflowEntity();
        order.setDocumentUrl(event.getVoucherUrl());
        order.getStateContext().setDocumentsUpdatedAt(Instant.now());
    }

    @HandleEvent
    public void handleVoucherCreated(TVoucherRecreated event, StateContext<EOrderState, GenericOrder> context) {
        genericWorkflowService.handleVoucherRecreated(event, context);
    }
}
