package ru.yandex.travel.orders.workflows.order.generic.handlers;

import java.time.Instant;
import java.util.Map;
import java.util.NoSuchElementException;
import java.util.Optional;
import java.util.UUID;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TPrice;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.GenericOrderUserRefund;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.MoneyRefund;
import ru.yandex.travel.orders.entities.MoneyRefundState;
import ru.yandex.travel.orders.entities.OrderRefund;
import ru.yandex.travel.orders.entities.OrderRefundPayload;
import ru.yandex.travel.orders.entities.ServiceRefundState;
import ru.yandex.travel.orders.entities.context.OrderItemContextState;
import ru.yandex.travel.orders.entities.promo.PromoCodeHelpers;
import ru.yandex.travel.orders.management.StarTrekService;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.proto.EOrderRefundType;
import ru.yandex.travel.orders.repository.MoneyRefundRepository;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.services.buses.BusNotificationHelper;
import ru.yandex.travel.orders.services.orders.GenericOrderMoneyRefundService;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.promo.UserOrderCounterService;
import ru.yandex.travel.orders.services.train.TrainRefundLogService;
import ru.yandex.travel.orders.workflow.invoice.proto.TMoneyMarkup;
import ru.yandex.travel.orders.workflow.invoice.proto.TScheduleClearing;
import ru.yandex.travel.orders.workflow.notification.proto.TNotificationComplete;
import ru.yandex.travel.orders.workflow.notification.proto.TSend;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInsuranceAutoRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceNotRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TManualRefund;
import ru.yandex.travel.orders.workflow.order.proto.TServiceRefundFailed;
import ru.yandex.travel.orders.workflow.order.proto.TServiceRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TStartReservationCancellation;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.train.proto.TServiceOfficeRefunded;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherCreated;
import ru.yandex.travel.orders.workflow.voucher.proto.TVoucherRecreated;
import ru.yandex.travel.orders.workflows.order.OrderUtils;
import ru.yandex.travel.orders.workflows.order.generic.GenericWorkflowService;
import ru.yandex.travel.workflow.EWorkflowState;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@RequiredArgsConstructor
@Slf4j
public class RefundingStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderState, GenericOrder> {
    private final MoneyRefundRepository moneyRefundRepository;
    private final GenericOrderMoneyRefundService moneyRefundService;
    private final NotificationHelper notificationHelper;
    private final BusNotificationHelper busNotificationHelper;
    private final TrainRefundLogService trainRefundLogService;
    private final StarTrekService starTrekService;
    private final GenericWorkflowService genericWorkflowService;
    private final UserOrderCounterService counterService;

    @HandleEvent
    public void handleStartCancellation(TStartReservationCancellation event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        order.setUserActionScheduled(false);
    }

    @HandleEvent
    public void handle(TServiceOfficeRefunded event, StateContext<EOrderState, GenericOrder> ctx) {
        UUID itemId = UUID.fromString(event.getServiceId());
        UUID refundId = UUID.fromString(event.getOrderRefundId());
        handleServiceRefunded(itemId, refundId, event.getTargetFiscalItemsMap(),
                event.getTargetFiscalItemsMarkupMap(), event.getServiceConfirmed(), ctx);
    }

    @HandleEvent
    public void handle(TServiceRefunded event, StateContext<EOrderState, GenericOrder> ctx) {
        UUID itemId = UUID.fromString(event.getServiceId());
        UUID refundId = UUID.fromString(event.getOrderRefundId());
        handleServiceRefunded(itemId, refundId, event.getTargetFiscalItemsMap(),
                event.getTargetFiscalItemsMarkupMap(), event.getServiceConfirmed(), ctx);
    }

    private void handleServiceRefunded(UUID itemId, UUID refundId,
                                       Map<Long, TPrice> targetFiscalItemsMap,
                                       Map<Long, TMoneyMarkup> targetFiscalItemsMarkup,
                                       Boolean serviceConfirmed,
                                       StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        var itemState = order.getStateContext().getItem(itemId);
        if (serviceConfirmed) {
            itemState.setState(EOrderItemState.IS_CONFIRMED);
        } else {
            itemState.setState(EOrderItemState.IS_REFUNDED);
        }

        MoneyRefund moneyRefund = findMoneyRefundOrNull(order, refundId);
        if (moneyRefund == null) {
            moneyRefund = MoneyRefund.createPendingRefundFromProto(order, targetFiscalItemsMap,
                    targetFiscalItemsMarkup, refundId, "Refund");
            moneyRefundRepository.save(moneyRefund);
        } else {
            Preconditions.checkState(moneyRefund.getState() == MoneyRefundState.PENDING,
                    "Can combine fiscal items for the refund only while it's still waiting for its services but got %s",
                    moneyRefund.getState());
            moneyRefund.addTargetFiscalItems(targetFiscalItemsMap, targetFiscalItemsMarkup);
        }
        OrderRefund orderRefund = findOrderRefundOrThrow(order, refundId);
        // TODO(ganintsev, tlg-13): remove this type check (TRAVELBACK-1668)
        if (orderRefund instanceof GenericOrderUserRefund) {
            OrderRefundPayload refundPayload = ((GenericOrderUserRefund) orderRefund).getPayload();
            refundPayload.changeServiceRefundState(itemId, ServiceRefundState.REFUNDING, ServiceRefundState.SUCCESS);
            UUID nextPendingServiceRefundId = refundPayload.nextPendingServiceRefund();
            if (nextPendingServiceRefundId != null) {
                genericWorkflowService.scheduleServiceRefund(order, (GenericOrderUserRefund) orderRefund,
                        genericWorkflowService.getRefundToken(refundPayload.getRefundToken()),
                        nextPendingServiceRefundId, ctx);
                return;
            } else if (order.getStateContext().anyServiceRefunding()) {
                return;
            }
            Preconditions.checkState(refundPayload.allServiceRefundsDone());
            order.getStateContext().setRefundsUpdatedAt(Instant.now());
            handleAllServicesRefunded(order, ((GenericOrderUserRefund) orderRefund), moneyRefund, ctx);
        } else {
            if (order.getStateContext().anyServiceRefunding()) {
                return;
            }
            orderRefund.setState(EOrderRefundState.RS_WAITING_INVOICE_REFUND);
            order.getStateContext().setRefundsUpdatedAt(Instant.now());
            if (!moneyRefundService.hasActiveMoneyRefunds(order)) {
                moneyRefundService.startInvoiceRefund(moneyRefund, ctx);
            }
        }
    }

    private MoneyRefund findMoneyRefundOrNull(GenericOrder order, UUID orderRefundId) {
        return order.getMoneyRefunds().stream()
                .filter(r -> orderRefundId.equals(r.getOrderRefundId()))
                .findFirst()
                .orElse(null);
    }

    private OrderRefund findOrderRefundOrThrow(GenericOrder order, UUID orderRefundId) {
        return order.getOrderRefunds().stream()
                .filter(x -> x.getId().equals(orderRefundId))
                .findFirst().orElseThrow();
    }

    private void handleAllServicesRefunded(GenericOrder order,
                                           GenericOrderUserRefund orderRefund,
                                           MoneyRefund moneyRefund,
                                           StateContext<EOrderState, GenericOrder> ctx) {
        if (orderRefund.getPayload().isAnyServiceRefundSuccess()) {
            Preconditions.checkArgument(moneyRefund != null, "Money refund can not be null for success service refund");
            orderRefund.setState(EOrderRefundState.RS_WAITING_INVOICE_REFUND);
            order.getStateContext().setRefundsUpdatedAt(Instant.now());
            if (!moneyRefundService.hasActiveMoneyRefunds(order)) {
                moneyRefundService.startInvoiceRefund(moneyRefund, ctx);
            }
        } else {
            orderRefund.setState(EOrderRefundState.RS_FAILED);
            ctx.setState(EOrderState.OS_CONFIRMED);
        }
    }

    @HandleEvent
    public void handle(TServiceRefundFailed event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        var refundId = UUID.fromString(event.getOrderRefundId());
        var itemId = UUID.fromString(event.getServiceId());
        OrderRefund orderRefund = findOrderRefundOrThrow(order, refundId);
        OrderItemContextState itemState = order.getStateContext().getItem(itemId);
        itemState.setState(EOrderItemState.IS_CONFIRMED);
        if (orderRefund instanceof GenericOrderUserRefund) {
            OrderRefundPayload refundPayload = ((GenericOrderUserRefund) orderRefund).getPayload();
            refundPayload.changeServiceRefundState(itemId, ServiceRefundState.REFUNDING, ServiceRefundState.FAILED);
            UUID nextPendingServiceRefundId = refundPayload.nextPendingServiceRefund();
            if (nextPendingServiceRefundId != null) {
                genericWorkflowService.scheduleServiceRefund(order, (GenericOrderUserRefund) orderRefund,
                        genericWorkflowService.getRefundToken(refundPayload.getRefundToken()),
                        nextPendingServiceRefundId, ctx);
                return;
            } else if (order.getStateContext().anyServiceRefunding()) {
                return;
            }
            Preconditions.checkState(refundPayload.allServiceRefundsDone());
            order.getStateContext().setRefundsUpdatedAt(Instant.now());
            handleAllServicesRefunded(order, ((GenericOrderUserRefund) orderRefund),
                    findMoneyRefundOrNull(order, refundId), ctx);
        } else {
            throw new RuntimeException("Can not handle failed refund " + refundId);
        }
    }

    @HandleEvent
    public void handle(TInvoiceRefunded event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        MoneyRefund moneyRefund = order.getMoneyRefunds().stream()
                .filter(r -> r.getState() == MoneyRefundState.IN_PROGRESS)
                .findFirst()
                .orElseThrow(() -> new IllegalStateException("No active MoneyRefund object"));
        moneyRefund.setState(MoneyRefundState.REFUNDED);

        order.getStateContext().setRefundsUpdatedAt(Instant.now());
        Invoice currentInvoice = ctx.getWorkflowEntity().getCurrentInvoice();

        Preconditions.checkState(currentInvoice != null, "Current invoice must be present");
        Preconditions.checkArgument(currentInvoice.getId().equals(UUID.fromString(event.getInvoiceId())),
                "Current invoice %s, got invoice id %s in event", currentInvoice.getId(), event.getInvoiceId());

        // There is no OrderRefund for manual money refund
        if (!Strings.isNullOrEmpty(event.getOrderRefundId())) {
            var refundId = UUID.fromString(event.getOrderRefundId());
            OrderRefund orderRefund = order.getOrderRefunds().stream()
                    .filter(x -> x.getId().equals(refundId)).findAny()
                    .orElseThrow(() -> new NoSuchElementException("No such refund: " + refundId));
            orderRefund.setInvoiceRefundType(event.getRefundType());
            orderRefund.setState(EOrderRefundState.RS_REFUNDED);

            if (OrderCompatibilityUtils.isTrainOrder(order)) {
                switch (orderRefund.getRefundType()) {
                    case RT_TRAIN_OFFICE_REFUND:
                    case RT_GENERIC_USER_REFUND:
                        trainRefundLogService.logRefund(order, OrderUtils.getSuccessfulTicketRefunds(orderRefund));
                        break;
                }

                switch (orderRefund.getRefundType()) {
                    case RT_TRAIN_OFFICE_REFUND:
                    case RT_GENERIC_USER_REFUND:
                        var emailWorkflowId = notificationHelper.createWorkflowForTrainRefundEmailV2(order,
                                orderRefund, OrderUtils.getTrainTicketRefunds(orderRefund));
                        var sendEvent = TSend.newBuilder().build();
                        ctx.scheduleExternalEvent(emailWorkflowId, sendEvent);
                        break;
                    case RT_TRAIN_INSURANCE_AUTO_RETURN:
                        Preconditions.checkState(moneyRefundService.getPendingMoneyRefund(order) == null,
                                "Train insurance auto refund must contain only one money refund");
                        ctx.setState(EOrderState.OS_WAITING_CONFIRMATION);
                        ctx.scheduleEvent(TInsuranceAutoRefunded.newBuilder().build());
                        return;
                    default:
                        throw new UnsupportedOperationException(String.format("Unsupported refund type %s (id=%s)",
                                orderRefund.getRefundType(), orderRefund.getId()));
                }
            } else if (OrderCompatibilityUtils.isBusOrder(order)) {
                Preconditions.checkState(orderRefund.getRefundType() == EOrderRefundType.RT_GENERIC_USER_REFUND);
                var emailWorkflowId = busNotificationHelper.createWorkflowForRefundEmail(order,
                        (GenericOrderUserRefund) orderRefund);
                var sendEvent = TSend.newBuilder().build();
                ctx.scheduleExternalEvent(emailWorkflowId, sendEvent);
            }
        }
        // TODO (mbobrov, ganintsev): think of scheduling it further after refund
        ctx.scheduleExternalEvent(currentInvoice.getWorkflow().getId(),
                TScheduleClearing.newBuilder().setClearAt(ProtoUtils.fromInstant(Instant.now())).build());

        MoneyRefund pendingRefund = moneyRefundService.getPendingMoneyRefund(order);
        if (pendingRefund != null) {
            moneyRefundService.startInvoiceRefund(pendingRefund, ctx);
        } else {
            if (order.getStateContext().allItemsCancelled()) {
                ctx.setState(EOrderState.OS_CANCELLED);
            } else if (order.getStateContext().allItemsRefunded()) {
                ctx.setState(EOrderState.OS_REFUNDED);
                counterService.onOrderRefunded(order);
            } else {
                ctx.setState(EOrderState.OS_CONFIRMED);
            }
        }

        PromoCodeHelpers.blacklistGeneratedPromoCodesForOrder(order);
    }

    @HandleEvent
    public void handleManualMoneyRefund(TManualRefund event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        MoneyRefund moneyRefund = MoneyRefund.createPendingRefundFromProto(order, event.getTargetFiscalItemsMap(),
                event.getTargetFiscalItemsMarkupMap(), null, event.getReason());
        moneyRefundRepository.save(moneyRefund);
        if (!moneyRefundService.hasActiveMoneyRefunds(order)) {
            moneyRefundService.startInvoiceRefund(moneyRefund, ctx);
        }
    }

    @HandleEvent
    public void handleInvoiceNotRefunded(TInvoiceNotRefunded event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        order.getWorkflow().setState(EWorkflowState.WS_PAUSED);
        starTrekService.createIssueForInvoiceNotRefunded(ctx.getWorkflowEntity(),
                ProtoUtils.fromStringOrNull(event.getOrderRefundId()),
                ProtoUtils.fromStringOrNull(event.getInvoiceId()), ctx,
                "refund failed");
    }

    @HandleEvent
    public void handle(TClearingInProcess event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();

        Optional<MoneyRefund> pendingRefund = order.getMoneyRefunds().stream()
                .filter(r -> r.getState() == MoneyRefundState.IN_PROGRESS).findFirst();
        Preconditions.checkState(pendingRefund.isPresent(), "Pending refund must be present");
        pendingRefund.get().setState(MoneyRefundState.WAITING_CLEARING);
    }

    @HandleEvent
    public void handle(TInvoiceCleared event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();

        Optional<MoneyRefund> pendingRefund = order.getMoneyRefunds().stream()
                .filter(r -> r.getState() == MoneyRefundState.WAITING_CLEARING).findFirst();

        pendingRefund.ifPresent(r -> {
            r.setState(MoneyRefundState.PENDING);
            moneyRefundService.startInvoiceRefund(r, ctx);
        });
    }

    @HandleEvent
    public void handleVoucherCreated(TVoucherCreated event, StateContext<EOrderState, GenericOrder> context) {
        genericWorkflowService.handleVoucherCreated(event, context);
    }

    @HandleEvent
    public void handleVoucherCreated(TVoucherRecreated event, StateContext<EOrderState, GenericOrder> context) {
        genericWorkflowService.handleVoucherRecreated(event, context);
    }

    @HandleEvent
    public void handleNotificationComplete(TNotificationComplete event,
                                           StateContext<EOrderState, GenericOrder> context) {
        log.info("Confirmation notification sent for order {}", context.getWorkflowEntity().getId());
    }
}
