package ru.yandex.travel.orders.workflows.order.generic.handlers;

import java.util.List;
import java.util.UUID;

import com.google.common.base.Preconditions;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.order.proto.TCheckoutStart;
import ru.yandex.travel.orders.workflow.order.proto.TServiceAdded;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TStartReservationCancellation;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.TReservationStart;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TInsuranceReservationStart;
import ru.yandex.travel.orders.workflow.train.proto.TAddInsurance;
import ru.yandex.travel.orders.workflows.order.generic.GenericWorkflowService;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;

@RequiredArgsConstructor
@Slf4j
public class ReservedStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderState, GenericOrder> {
    private final GenericWorkflowService genericWorkflowService;

    @HandleEvent
    public void handleServiceAdded(TServiceAdded event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        order.setUserActionScheduled(false);

        OrderItem newService = order.getOrderItems().stream()
                .filter(oi -> oi.getId().equals(UUID.fromString(event.getServiceId())))
                .findAny().orElseThrow(() -> new IllegalArgumentException("No such item: " + event.getServiceId()));
        order.getStateContext().serviceAdded(newService);
        order.getStateContext().getItem(newService.getId()).setState(EOrderItemState.IS_RESERVING);

        ctx.setState(EOrderState.OS_WAITING_RESERVATION);
        ctx.scheduleExternalEvent(newService.getWorkflow().getId(), TReservationStart.newBuilder().build());
    }

    @HandleEvent
    public void handleAddInsurance(TAddInsurance event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        Preconditions.checkState(OrderCompatibilityUtils.isTrainOrder(order), "There is no train order to add insurance");
        order.toggleUserActionScheduled(false);
        List<TrainOrderItem> trainOrderItems = OrderCompatibilityUtils.getTrainOrderItems(order);
        for (TrainOrderItem orderItem : trainOrderItems) {
            order.getStateContext().getItem(orderItem.getId()).setState(EOrderItemState.IS_RESERVING);
            ctx.scheduleExternalEvent(orderItem.getWorkflow().getId(), TInsuranceReservationStart.newBuilder().build());
        }
        order.getStateContext().setTrainInsuranceAdded(true);
        ctx.setState(EOrderState.OS_WAITING_RESERVATION);
    }

    @HandleEvent
    public void handleCheckout(TCheckoutStart event, StateContext<EOrderState, GenericOrder> ctx) {
        log.info("The order has been completely formed, staring payment process");
        ctx.setState(EOrderState.OS_WAITING_PAYMENT);
        ctx.getWorkflowEntity().setUserActionScheduled(false);
        ctx.getWorkflowEntity().getStateContext().setCheckedOut(true);
    }

    @HandleEvent
    public void handleCancelled(TServiceCancelled event, StateContext<EOrderState, GenericOrder> ctx) {
        genericWorkflowService.handleServiceCancelled(event, ctx);
    }

    @HandleEvent
    public void handleStartReservationCancellation(TStartReservationCancellation event,
                                                   StateContext<EOrderState, GenericOrder> ctx) {
        genericWorkflowService.handleStartReservationCancellation(event, ctx);
    }
}
