package ru.yandex.travel.orders.workflows.order.generic.handlers;

import java.util.List;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.context.OrderItemContextState;
import ru.yandex.travel.orders.services.promo.PromoCodeApplicationService;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.order.proto.TClearingInProcess;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquireErrorOccurred;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceReserved;
import ru.yandex.travel.orders.workflow.order.proto.TStartReservationCancellation;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.TCancellationStart;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TInsuranceReservationStart;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

import static java.util.stream.Collectors.toList;

@Slf4j
@IgnoreEvents(types = {
        TInvoicePaymentStarted.class,
        TStartReservationCancellation.class,
        TMoneyAcquireErrorOccurred.class,
        TInsuranceReservationStart.class,
        TInvoiceCleared.class,
        TClearingInProcess.class
})
@RequiredArgsConstructor
public class WaitingCancellationStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<EOrderState, GenericOrder> {
    private final PromoCodeApplicationService promoCodeApplicationService;

    @HandleEvent
    public void handleMoneyAcquired(TMoneyAcquired event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        order.getStateContext().setMoneyAcquired(true);
        var currentInvoice = order.getCurrentInvoice();
        ctx.scheduleExternalEvent(currentInvoice.getWorkflow().getId(), InvoiceUtils.buildFullRefund(currentInvoice));
    }

    @HandleEvent
    public void handleReserved(TServiceReserved event, StateContext<EOrderState, GenericOrder> ctx) {
        var order = ctx.getWorkflowEntity();
        OrderItemContextState itemState = order.getStateContext().getItem(UUID.fromString(event.getServiceId()));
        itemState.setState(EOrderItemState.IS_CANCELLING);
        ctx.scheduleExternalEvent(itemState.getWorkflowId(), TCancellationStart.getDefaultInstance());
    }

    @HandleEvent
    public void handleInvoiceRefunded(TInvoiceRefunded event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        order.getStateContext().setMoneyAcquired(false);
        checkCancelled(order, ctx);
    }

    @HandleEvent
    public void handleServiceCancelled(TServiceCancelled event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        UUID serviceId = UUID.fromString(event.getServiceId());
        order.getStateContext().serviceCancelled(serviceId);
        checkCancelled(order, ctx);
    }

    private void checkCancelled(GenericOrder order, StateContext<EOrderState, GenericOrder> ctx) {
        if (order.getStateContext().allItemsCancelled() && !order.getStateContext().isMoneyAcquired()) {
            ctx.setState(EOrderState.OS_CANCELLED);
            promoCodeApplicationService.freePromoCodeActivations(order);
        } else if (!order.getStateContext().allItemsCancelled()) {
            List<EOrderItemState> states = order.getStateContext().getOrderItems().stream()
                    .map(OrderItemContextState::getState)
                    .collect(toList());
            log.info("Not all order items are cancelled yet, waiting for the rest; states: {}", states);
        } else {
            log.info("Acquired money aren't returned yet, keep waiting");
        }
    }
}
