package ru.yandex.travel.orders.workflows.order.generic.handlers;

import java.util.List;
import java.util.UUID;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.orders.commons.proto.ECancellationReason;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.Invoice;
import ru.yandex.travel.orders.entities.Order;
import ru.yandex.travel.orders.entities.OrderItem;
import ru.yandex.travel.orders.entities.TrainOrderInsuranceAutoRefund;
import ru.yandex.travel.orders.entities.TrainOrderItem;
import ru.yandex.travel.orders.entities.TrustInvoice;
import ru.yandex.travel.orders.entities.context.OrderItemContextState;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.proto.EOrderRefundType;
import ru.yandex.travel.orders.services.NotificationHelper;
import ru.yandex.travel.orders.services.finances.FinancialEventService;
import ru.yandex.travel.orders.services.orders.OrderCompatibilityUtils;
import ru.yandex.travel.orders.services.promo.UserOrderCounterService;
import ru.yandex.travel.orders.services.promo.mir2020.Mir2020PromoService;
import ru.yandex.travel.orders.services.promo.taxi2020.Taxi2020PromoService;
import ru.yandex.travel.orders.services.suburban.SuburbanNotificationHelper;
import ru.yandex.travel.orders.workflow.notification.proto.TSend;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.order.proto.TInsuranceAutoRefunded;
import ru.yandex.travel.orders.workflow.order.proto.TInvoiceCleared;
import ru.yandex.travel.orders.workflow.order.proto.TRenderDocuments;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceConfirmed;
import ru.yandex.travel.orders.workflow.order.proto.TStartReservationCancellation;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.TCancellationStart;
import ru.yandex.travel.orders.workflow.orderitem.train.proto.TInsuranceRefundStart;
import ru.yandex.travel.orders.workflow.train.proto.TServiceConfirmedWithoutInsurance;
import ru.yandex.travel.orders.workflows.invoice.trust.InvoiceUtils;
import ru.yandex.travel.orders.workflows.invoice.trust.jobs.ClearingService;
import ru.yandex.travel.orders.workflows.order.OrderUtils;
import ru.yandex.travel.orders.workflows.order.train.TrainWorkflowService;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@IgnoreEvents(types = TInvoiceCleared.class)
@RequiredArgsConstructor
@Slf4j
public class WaitingConfirmationStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderState, GenericOrder> {
    private final FinancialEventService financialEventService;
    private final Taxi2020PromoService taxi2020PromoService;
    private final Mir2020PromoService mir2020PromoService;
    private final NotificationHelper notificationHelper;
    private final SuburbanNotificationHelper suburbanNotificationHelper;
    private final TrainWorkflowService trainWorkflowService;
    private final UserOrderCounterService userOrderCounterService;
    private final ClearingService clearingService;

    @HandleEvent
    public void handleStartCancellation(TStartReservationCancellation event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        order.setUserActionScheduled(false);
    }

    @HandleEvent
    public void handleInsuranceAutoRefunded(TInsuranceAutoRefunded event, StateContext<EOrderState, GenericOrder> ctx) {
        onAllItemsConfirmed(ctx);
    }

    @HandleEvent
    public void onConfirmation(TServiceConfirmed event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        UUID serviceId = UUID.fromString(event.getServiceId());
        order.getStateContext().serviceConfirmed(serviceId);
        if (!order.getStateContext().allItemsConfirmed()) {
            log.info("Not all order items are completed yet, waiting for the rest");
            return;
        }

        if (order.getStateContext().isTrainInsuranceAutoReturnPending()) {
            order.getStateContext().setTrainInsuranceAutoReturnPending(false);
            onAllItemsConfirmedWithInsuranceRefund(ctx);
            return;
        }
        onAllItemsConfirmed(ctx);
    }

    @HandleEvent
    public void handleServiceConfirmedWithoutInsurance(TServiceConfirmedWithoutInsurance event,
                                                       StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        UUID serviceId = UUID.fromString(event.getServiceId());
        order.getStateContext().serviceConfirmed(serviceId);
        order.getStateContext().setTrainInsuranceAutoReturnPending(true);

        if (!order.getStateContext().allItemsConfirmed()) {
            log.info("Not all order items are completed yet, waiting for the rest");
            return;
        }

        if (order.getStateContext().isTrainInsuranceAutoReturnPending()) {
            order.getStateContext().setTrainInsuranceAutoReturnPending(false);
            onAllItemsConfirmedWithInsuranceRefund(ctx);
            return;
        }
        onAllItemsConfirmed(ctx);
    }

    private TrainOrderInsuranceAutoRefund getTrainOrderInsuranceAutoRefund(GenericOrder order) {
        return (TrainOrderInsuranceAutoRefund) order.getOrderRefunds().stream()
                .filter(x -> x.getRefundType() == EOrderRefundType.RT_TRAIN_INSURANCE_AUTO_RETURN)
                .findFirst().orElse(null);
    }

    private void onAllItemsConfirmed(StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();

        for (OrderItem orderItem : order.getOrderItems()) {
            financialEventService.registerConfirmedService(orderItem);
        }

        userOrderCounterService.onOrderConfirmed(order.getId(), order.getDisplayType());

        if (OrderCompatibilityUtils.isHotelOrder(order)) {
            taxi2020PromoService.registerConfirmedOrder(order);
            mir2020PromoService.registerConfirmedOrder(order);
        } else if (OrderCompatibilityUtils.isTrainOrder(order)) {
            trainWorkflowService.createPromoCodeForHotels(order);
        }

        ctx.setState(EOrderState.OS_CONFIRMED);
        clearInvoicesIfNecessary(ctx);
        scheduleNotifications(ctx);
    }

    private void onAllItemsConfirmedWithInsuranceRefund(StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        var refund = TrainOrderInsuranceAutoRefund.createForOrder(order);
        List<TrainOrderItem> trainInsuranceItemsToRefund = OrderCompatibilityUtils.getTrainOrderItems(order)
                .stream().filter(x -> x.getPayload().getInsuranceStatus() == InsuranceStatus.CHECKED_OUT)
                .collect(Collectors.toList());
        for (var item : trainInsuranceItemsToRefund) {
            OrderItemContextState itemState = order.getStateContext().getItem(item.getId());
            itemState.setState(EOrderItemState.IS_REFUNDING);
            ctx.scheduleExternalEvent(itemState.getWorkflowId(), TInsuranceRefundStart.newBuilder()
                    .setOrderRefundId(refund.getId().toString()).build());
        }
        refund.setState(EOrderRefundState.RS_WAITING_SERVICE_REFUND);
        ctx.setState(EOrderState.OS_REFUNDING);
    }

    private void clearInvoicesIfNecessary(StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        TrustInvoice invoice = OrderUtils.getRequiredCurrentInvoice(order);
        if (order.getMoneyRefunds().size() > 0) {
            clearingService.clearInvoice(invoice);
        } else if (OrderCompatibilityUtils.isTrainOrder(order)) {
            List<TrainOrderItem> trainOrderItems = OrderCompatibilityUtils.getTrainOrderItems(order);
            if (trainOrderItems.stream().anyMatch(x -> x.getPayload().getInsuranceStatus() == InsuranceStatus.CHECKED_OUT)) {
                clearingService.clearInvoice(invoice);
            }
        }
    }

    private void scheduleNotifications(StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        if (OrderCompatibilityUtils.isTrainOrder(order)) {
            scheduleTrainNotifications(ctx);
        } else if (OrderCompatibilityUtils.isSuburbanOrder(order)) {
            scheduleSuburbanNotifications(ctx);
        } else if (OrderCompatibilityUtils.isBusOrder(order)) {
            ctx.scheduleEvent(TRenderDocuments.newBuilder().build());
        }
    }

    private void scheduleTrainNotifications(StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        TrainOrderInsuranceAutoRefund insuranceAutoReturn = order.getOrderRefunds().stream()
                .filter(x -> x.getRefundType() == EOrderRefundType.RT_TRAIN_INSURANCE_AUTO_RETURN &&
                        x.getState() == EOrderRefundState.RS_REFUNDED)
                .map(x -> (TrainOrderInsuranceAutoRefund) x)
                .findFirst().orElse(null);
        var emailWorkflowId = notificationHelper.createWorkflowForSuccessfulTrainEmailV2(order,
                insuranceAutoReturn);
        var sendEvent = TSend.newBuilder().build();
        ctx.scheduleExternalEvent(emailWorkflowId, sendEvent);

        try {
            var smsWorkflowId = notificationHelper.createWorkflowForSuccessfulTrainSms(order);
            ctx.scheduleExternalEvent(smsWorkflowId, sendEvent);
        } catch (Exception e) {
            log.error("Unable to create workflow for successful train SMS. SMS won't be sent", e);
        }

        trainWorkflowService.scheduleTrainHotelPromoMail(order);
    }

    private void scheduleSuburbanNotifications(StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        var emailWorkflowId = suburbanNotificationHelper.createWorkflowForSuburbanOrderConfirmedEmail(order);
        ctx.scheduleExternalEvent(emailWorkflowId, TSend.getDefaultInstance());
    }

    @HandleEvent
    public void handleServiceCancelled(TServiceCancelled event, StateContext<EOrderState, GenericOrder> ctx) {
        Order order = ctx.getWorkflowEntity();

        UUID serviceId = UUID.fromString(event.getServiceId());
        OrderItemContextState item = order.getStateContext().getItem(serviceId);
        item.setState(EOrderItemState.IS_CANCELLED);
        // schedule full refund for all invoice items
        Invoice invoice = OrderUtils.getRequiredCurrentInvoice(order);
        ctx.scheduleExternalEvent(invoice.getWorkflow().getId(), InvoiceUtils.buildFullRefund(invoice));
        order.getStateContext().serviceCancelled(UUID.fromString(event.getServiceId()));
        for (OrderItemContextState itemState : order.getStateContext().getOrderItems()) {
            if (itemState.getState() != EOrderItemState.IS_CANCELLED) {
                itemState.setState(EOrderItemState.IS_CANCELLING);
                ctx.scheduleExternalEvent(itemState.getWorkflowId(), TCancellationStart.getDefaultInstance());
            }
        }
        order.getStateContext().setCancellationReason(ECancellationReason.CR_CONFIRMATION_FAILED);
        ctx.setState(EOrderState.OS_WAITING_CANCELLATION);
    }
}
