package ru.yandex.travel.orders.workflows.order.generic.handlers;

import java.text.MessageFormat;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;

import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.context.OrderItemContextState;
import ru.yandex.travel.orders.services.AccountService;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquireErrorOccurred;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TStartReservationCancellation;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.TConfirmationStart;
import ru.yandex.travel.orders.workflows.order.generic.GenericWorkflowService;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

@Slf4j
@RequiredArgsConstructor
@IgnoreEvents(types = {
        TInvoicePaymentStarted.class,
        TMoneyAcquireErrorOccurred.class
})
public class WaitingPaymentStateHandler extends AnnotatedStatefulWorkflowEventHandler<EOrderState, GenericOrder> {
    private final AccountService accountService;
    private final GenericWorkflowService genericWorkflowService;

    @HandleEvent
    public void handleMoneyAcquired(TMoneyAcquired event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        order.getStateContext().setMoneyAcquired(true);
        // todo(mbobrov,tlg-13): remove the code below when the payment blackbox is integrated here
        Money totalCost = order.calculateTotalCost();
        Money acquired = this.accountService.getAccountBalance(order.getAccount().getId());
        if (!totalCost.isEqualTo(acquired)) {
            throw new RuntimeException(
                    MessageFormat.format("Money acquired for order {0} does not equal to the expected total (acquired" +
                                    " {1}, expected {2})",
                            order.getId(), acquired, totalCost)
            );
        }
        // todo end
        for (OrderItemContextState itemState : order.getStateContext().getOrderItems()) {
            itemState.setState(EOrderItemState.IS_CONFIRMING);
            ctx.scheduleExternalEvent(itemState.getWorkflowId(), TConfirmationStart.newBuilder().build());
        }
        ctx.setState(EOrderState.OS_WAITING_CONFIRMATION);
    }

    @HandleEvent
    public void handleCancelled(TServiceCancelled event, StateContext<EOrderState, GenericOrder> ctx) {
        genericWorkflowService.handleServiceCancelled(event, ctx);
    }

    @HandleEvent
    public void handleStartReservationCancellation(TStartReservationCancellation event,
                                                   StateContext<EOrderState, GenericOrder> ctx) {
        genericWorkflowService.handleStartReservationCancellation(event, ctx);
    }
}
