package ru.yandex.travel.orders.workflows.order.generic.handlers;

import java.util.List;
import java.util.UUID;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;

import ru.yandex.travel.hotels.common.orders.CancellationDetails;
import ru.yandex.travel.orders.entities.GenericOrder;
import ru.yandex.travel.orders.entities.context.OrderItemContextState;
import ru.yandex.travel.orders.services.promo.PromoCodeApplicationService;
import ru.yandex.travel.orders.workflow.order.generic.proto.EOrderState;
import ru.yandex.travel.orders.workflow.order.proto.TInvoicePaymentStarted;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquireErrorOccurred;
import ru.yandex.travel.orders.workflow.order.proto.TMoneyAcquired;
import ru.yandex.travel.orders.workflow.order.proto.TServiceCancelled;
import ru.yandex.travel.orders.workflow.order.proto.TServiceReserved;
import ru.yandex.travel.orders.workflow.order.proto.TStartReservationCancellation;
import ru.yandex.travel.orders.workflow.orderitem.generic.proto.EOrderItemState;
import ru.yandex.travel.orders.workflows.order.generic.GenericWorkflowService;
import ru.yandex.travel.workflow.StateContext;
import ru.yandex.travel.workflow.base.AnnotatedStatefulWorkflowEventHandler;
import ru.yandex.travel.workflow.base.HandleEvent;
import ru.yandex.travel.workflow.base.IgnoreEvents;

import static java.util.stream.Collectors.toList;

@Slf4j
@RequiredArgsConstructor
@IgnoreEvents(types = {
        TInvoicePaymentStarted.class,
        TMoneyAcquireErrorOccurred.class
})
public class WaitingReservationStateHandler
        extends AnnotatedStatefulWorkflowEventHandler<EOrderState, GenericOrder> {
    private final PromoCodeApplicationService promoCodeApplicationService;
    private final GenericWorkflowService genericWorkflowService;

    @HandleEvent
    public void handleMoneyAcquired(TMoneyAcquired event, StateContext<EOrderState, GenericOrder> ctx) {
        GenericOrder order = ctx.getWorkflowEntity();
        order.getStateContext().setMoneyAcquired(true);
    }

    @HandleEvent
    public void handleReserved(TServiceReserved event, StateContext<EOrderState, GenericOrder> ctx) {
        var order = ctx.getWorkflowEntity();
        UUID serviceId = UUID.fromString(event.getServiceId());
        order.getStateContext().serviceReserved(serviceId);
        if (order.getStateContext().allItemsReserved()) {
            if (!order.getStateContext().isPromoCodesApplied()) {
                promoCodeApplicationService.usePromoCodeActivationsAndApplyDiscounts(order, true);
                order.getStateContext().setPromoCodesApplied(true);
            }
            if (!order.isTrainRebookingEnabled()) {
                order.refreshExpiresAt();
            }
            if (order.getStateContext().isCheckedOut()) {
                ctx.setState(EOrderState.OS_WAITING_PAYMENT);
                if (order.getStateContext().isMoneyAcquired()) {
                    ctx.scheduleEvent(TMoneyAcquired.newBuilder().build());
                }
            } else {
                ctx.setState(EOrderState.OS_RESERVED);
            }
            if (order.getStateContext().getPendingCancellation() == Boolean.TRUE) {
                order.getStateContext().setPendingCancellation(false);
                if (order.getStateContext().isMoneyAcquired()) {
                    log.info("Rejecting the pending cancellation operation as the payment has been completed");
                } else {
                    log.info("Resuming the pending cancellation operation");
                    ctx.scheduleEvent(TStartReservationCancellation.newBuilder()
                            .setReason(CancellationDetails.Reason.USER_INTENTION.toString()).build());
                }
            }
        } else {
            List<EOrderItemState> states = order.getStateContext().getOrderItems().stream()
                    .map(OrderItemContextState::getState)
                    .collect(toList());
            log.info("Some order items have not been reserved yet, waiting for the rest; states: {}", states);
        }
    }

    @HandleEvent
    public void handleCancelled(TServiceCancelled event, StateContext<EOrderState, GenericOrder> ctx) {
        genericWorkflowService.handleServiceCancelled(event, ctx);
    }

    @HandleEvent
    public void handleStartReservationCancellation(TStartReservationCancellation event,
                                                   StateContext<EOrderState, GenericOrder> ctx) {
        log.info("Reservation is in progress, will resume cancellation when it's completed");
        GenericOrder order = ctx.getWorkflowEntity();
        order.getStateContext().setPendingCancellation(true);
    }
}
